/**
 * SPDX-FileCopyrightText: 2022-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.6
import QtQuick.Window 2.1 as QtQuick
import Sailfish.Silica 1.0
import QtFeedback 5.0
import QtGraphicalEffects 1.0
import Sailfish.Policy 1.0
import MeeGo.QOfono 0.2
import MeeGo.Connman 0.2
import QtQuick.Layouts 1.1
import org.nemomobile.lipstick 0.1
import org.nemomobile.ofono 1.0
import com.jolla.settings.system 1.0
import Sailfish.Silica.private 1.0


Item {
    id: root

    readonly property color emergencyBackgroundPrimaryColor: "#480B0B"
    readonly property color emergencyBackgroundSecondaryColor: "#FF9898"
    readonly property color emergencyTextColor: "#FFFFFF"
    property string titleText
    property string enteredNumber
    // modem for emergency callsnumeric
    property string modemPath: modemManager.defaultVoiceModem || manager.defaultModem
    property QtObject _voiceCallManager
    readonly property bool isSmallTablet: LayoutClassProvider.currentDisplayClass & DisplayLayoutClass.SmallTablet
    readonly property bool isPortrait: QtQuick.Window.window.contentOrientation & Orientation.PortraitMask
    readonly property bool isLandscape: !isPortrait
    property bool isSmallAndLandscape: isSmallTablet && isLandscape
    readonly property  var indentForTablet: (Math.max(root.width, root.height)
                                             - simplePin.height - Theme.dp(56)
                                             - number.height - Theme.paddingMedium
                                             - bottomPanel.height
                                             - cancelButton.height - Theme.dp(40)) / 5

    signal canceled()

    function _dialEmergencyNumber() {
        if (!modem.online) {
            modem.onlineChanged.connect(_dialEmergencyNumber)
            // Disable flight mode on order to make an emergency call
            if (network.instance.offlineMode && AccessPolicy.flightModeToggleEnabled) {
                network.instance.offlineMode = false
            }

            modem.online = true
            return
        }

        modem.onlineChanged.disconnect(_dialEmergencyNumber)
        var emergencyNumbers = voiceCallManager.emergencyNumbers
        if (root.enteredNumber.length != 0 && emergencyNumbers.indexOf(root.enteredNumber) === -1) {
            enteredNumber = ""
            return
        }

        if (!_voiceCallManager) {
            _voiceCallManager = Qt.createQmlObject("import QtQuick 2.0; import org.nemomobile.voicecall 1.0; VoiceCallManager {}",
                                                   root, 'VoiceCallManager');
            _voiceCallManager.modemPath = Qt.binding(function() { return root.modemPath })
        }

        _voiceCallManager.dial(root.enteredNumber)
    }

    function clear() {
        enteredNumber = ""
        number.focus = false
    }

    anchors.fill: parent

    Rectangle {
        // emergency background
        anchors.fill: parent
        color: root.emergencyBackgroundPrimaryColor
    }

    Label {
        id: textInfo

        anchors {
            top: parent.top
            topMargin: isSmallAndLandscape ? Theme.dp(68) : Theme.dp(124)
            horizontalCenter: parent.horizontalCenter
        }

        //% "Emergency call"
        text: qsTrId("settings_pin-la-emergency_call")
        font.pixelSize: Theme.fontSizeLarge
        color: root.emergencyBackgroundSecondaryColor
    }

    TextField {
        id: number

        anchors {
            bottom: simplePin.top
            bottomMargin: Theme.paddingMedium
            horizontalCenter: parent.horizontalCenter
        }

        text: enteredNumber
        _cursorBlinkEnabled: false
        background: null
        enableSoftwareInputPanel: false
        font.pixelSize: Theme.fontSizeExtraLarge
        horizontalAlignment: TextField.AlignHCenter
        color: root.emergencyTextColor
        readOnly: enteredNumber.length == 0
        labelVisible: false
    }

    Item {
        id: simplePin

        anchors {
            bottom: bottomPanel.top
            bottomMargin: isSmallAndLandscape ? Theme.dp(32) : Theme.dp(56)
            horizontalCenter: parent.horizontalCenter
        }

        width: isSmallAndLandscape ? Theme.dp(460) : Theme.dp(496)
        height: isSmallAndLandscape ? Theme.dp(498) : Theme.dp(546)

        Rectangle {
            anchors.fill: parent
            opacity: 0.1
            color: root.emergencyTextColor
            radius: Theme.dp(48)
        }

        Grid {
            anchors.centerIn: parent
            width: simplePin.width - Theme.dp(40) * 2
            height: simplePin.height - Theme.dp(40) * 2
            columns: 3
            columnSpacing: Theme.dp(40)
            rowSpacing: Theme.paddingSmall

            Repeater {
                model: ListModel {
                    ListElement { value: "1" }
                    ListElement { value: "2" }
                    ListElement { value: "3" }
                    ListElement { value: "4" }
                    ListElement { value: "5" }
                    ListElement { value: "6" }
                    ListElement { value: "7" }
                    ListElement { value: "8" }
                    ListElement { value: "9" }
                    ListElement { value: "*" }
                    ListElement { value: "0" }
                    ListElement { value: "#" }
                }

                BackgroundItem {
                    id: key

                    property bool showDown: false

                    width: isSmallAndLandscape ? Theme.dp(100) : Theme.dp(112)
                    height: width
                    highlightedColor: "transparent"

                    onPressed: {
                        showDown = true
                        enteredNumber += model.value
                        number.focus = true
                        downTimer.start()
                    }

                    Label {
                        anchors.centerIn: parent
                        text: model.value
                        color: key.down ? root.emergencyBackgroundSecondaryColor : root.emergencyTextColor
                        font.pixelSize: Theme.fontSizeExtraLarge
                    }

                    Rectangle {
                        anchors.fill: parent
                        color: key.down ? root.emergencyBackgroundSecondaryColor : "transparent"
                        opacity: 0.1
                        radius: Theme.dp(24)
                    }

                    Timer {
                        id: downTimer

                        interval: 50

                        onTriggered: showDown = false
                    }
                }
            }
        }
    }

    Item {
        id: bottomPanel

        anchors {
            bottom: cancelButton.top
            bottomMargin: Screen._isTablet && isPortrait ? indentForTablet : Theme.paddingLarge
            horizontalCenter: parent.horizontalCenter
        }

        height: Theme.dp(112)
        width: Theme.dp(496)

        Rectangle {
            id: addButton

            anchors {
                verticalCenter: parent.verticalCenter
                left: parent.left
            }

            width: Theme.dp(92)
            height: width
            radius: width * 0.5
            opacity: 0.1
            color: root.emergencyTextColor
        }

        Icon {
            anchors.centerIn: addButton
            opacity: Theme.opacityHigh
            source: "image://theme/icon-splus-add"
            color: root.emergencyTextColor
        }

        Rectangle {
            id: callButton

            anchors {
                left: addButton.right
                leftMargin: Theme.dp(46)
                verticalCenter: parent.verticalCenter
            }

            width: Theme.dp(220)
            height: isSmallAndLandscape ? Theme.dp(100) : Theme.itemSizeLarge
            radius: height * 0.5
            opacity: enteredNumber.length == 0 ? Theme.opacityHigh : 0.8

            MouseArea {
                anchors.fill: parent
                enabled: enteredNumber.length != 0

                onClicked: _dialEmergencyNumber()
            }
        }

        Icon {
            anchors.centerIn: callButton
            opacity: enteredNumber.length == 0 ? Theme.opacityHigh : 1.0
            source: "image://theme/icon-splus-answer"
            color: "black"
        }

        Rectangle {
            id: dellButton

            anchors {
                left: callButton.right
                leftMargin: Theme.dp(46)
                verticalCenter: parent.verticalCenter
            }

            width: Theme.dp(92)
            height: width
            radius: width * 0.5
            opacity: 0.1
            color: root.emergencyTextColor

            MouseArea {
                anchors.fill: parent
                enabled: enteredNumber.length != 0

                onClicked: {
                    enteredNumber = enteredNumber.substring(0, enteredNumber.length - 1)

                    if (enteredNumber.length == 0) {
                        number.focus = false
                    }
                }
            }
        }

        Icon {
            anchors.centerIn: dellButton
            source: "image://theme/icon-splus-backspace-alt"
            opacity: enteredNumber.length == 0 ? Theme.opacityHigh : 1.0
            color: root.emergencyTextColor
        }
    }

    Button {
        id: cancelButton

        anchors {
            horizontalCenter: parent.horizontalCenter
            bottom: parent.bottom
            bottomMargin: isSmallAndLandscape ? Theme.dp(0) : Theme.dp(40)
        }

        width: Theme.buttonWidthLarge

        //: Close emergency call mode
        //% "Cancel"
        text: qsTrId("emergency-call-cancel-button")
        backgroundColor: "transparent"
        highlightBackgroundColor: "transparent"
        color: root.emergencyTextColor

        onClicked: {
            root.canceled()
            root.clear()
        } 
    }

    OfonoModemManager { id: modemManager }
    OfonoManager { id: manager }

    OfonoModem {
        id: modem
        modemPath: root.modemPath
    }

    OfonoVoiceCallManager {
        id: voiceCallManager
        modemPath: root.modemPath
    }

    NetworkManagerFactory { id: network }
}
