/*
 * SPDX-FileCopyrightText: 2013-2019 Jolla Ltd
 * SPDX-FileCopyrightText: 2020-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.6
import Sailfish.Silica 1.0
import Sailfish.Accounts 1.0
import com.jolla.settings 1.0

Item {
    property Item accountsPage
    property string title
    property var model

    // This property is here for compatibility with older apps
    property bool noAccountCreationFlowAfterDeletion: false

    property QtObject _settingsLoader
    property QtObject _settingsAgentRunner
    property QtObject _credentialsUpdater
    property alias accountCreationManager: accountCreationManager

    signal requestAccountRemoval(var accountId) // deletion from settings

    // TODO: The "action" argument is not used anywhere in the function, consider removing it
    //       Presumably, it was used to control the PageStackAction argument
    function _showSettings(providerName, accountId, credentialsNeedUpdate, action) {
        if (_settingsLoader != null) {
            _settingsLoader.destroy()
        }

        _settingsLoader = settingsLoaderComponent.createObject(accountsPage)
        _settingsLoader.finished.connect(function() {
            const settingsAgent = _accountSettingsAgent(providerName, accountId)
            SettingsUtils.openPage(settingsAgent.initialPage,
                                   { }, 
                                   credentialsNeedUpdate ? PageStackAction.Immediate : undefined)

            if (credentialsNeedUpdate) {
                if (_credentialsUpdater != null) {
                    _credentialsUpdater.destroy()
                }

                var credentialsUpdaterComponent = Qt.createComponent(Qt.resolvedUrl("AccountCredentialsUpdater.qml"))
                if (credentialsUpdaterComponent.status != Component.Ready) {
                    throw new Error("Unable to load AccountCredentialsUpdater.qml")
                }

                _credentialsUpdater = credentialsUpdaterComponent.createObject(accountsPage,
                                                                               {"accountManager": accountManager})
                _credentialsUpdater.credentialsUpdated.connect(function(updatedAccountId) {
                    if (SettingsUtils.isSettingsApplication && splitView.count === 3) {
                        splitView.pop()
                    }

                    const updatedSettingsAgent = _accountSettingsAgent(providerName, accountId)
                    SettingsUtils.openPage(updatedSettingsAgent.initialPage, {}, PageStackAction.Immediate)
                })

                _credentialsUpdater.pushCredentialsUpdatePage(accountId)
            }
        })

        _settingsLoader.start(accountId)
    }

    function _accountSettingsAgent(providerName, accountId) {
        var accountRef = model.getByAccount(accountId)
        var agentProperties = {
            "accountManager": accountManager,
            "accountProvider": accountManager.provider(providerName),
            "accountsHeaderText": title,
            "accountId": accountId,
            "accountIsReadOnly": accountRef.accountReadOnly,
            "accountIsLimited": accountRef.accountLimited,
            "accountIsProvisioned": accountRef.accountProvisioned,
            "accountNotSignedIn": (accountRef.accountError === AccountModel.AccountNotSignedInError)
        }

        if (noAccountCreationFlowAfterDeletion) {
            agentProperties["noAccountCreationFlowAfterDeletion"] = true
        }

        if (agentProperties["accountProvider"] == null) {
            throw new Error("Unable to obtain provider with name: " + providerName)
        }

        var runnerProperties = {
            "agentComponentFileName": "/usr/share/accounts/ui/" + providerName + "-settings.qml",
            "agentProperties": agentProperties
        }

        var agentRunnerComponent = Qt.createComponent(Qt.resolvedUrl("AccountAgentRunner.qml"))
        if (agentRunnerComponent.status != Component.Ready) {
            throw new Error("Unable to load AccountAgentRunner.qml")
        }

        if (_settingsAgentRunner != null) {
            _settingsAgentRunner.destroy()
        }

        _settingsAgentRunner = agentRunnerComponent.createObject(accountCreationManager, runnerProperties)
        _settingsAgentRunner.finished.connect(function() {
            _settingsAgentRunner.destroy()
            _settingsAgentRunner = null
        })

        _settingsAgentRunner.agent.accountDeletionRequested.connect(function() {
            if (!noAccountCreationFlowAfterDeletion) {
                if (!SettingsUtils.isSettingsApplication || splitView.count < 2 || currentAccountId != -1) {
                    accountCreationManager.startAccountCreation()
                }
            }

            requestAccountRemoval(accountId)
        })

        _settingsAgentRunner.agent.accountCreationManager = accountCreationManager

        return _settingsAgentRunner.agent
    }

    AccountCreationManager {
        id: accountCreationManager

        endDestination: accountsPage
        _accountManager: accountManager
    }

    AccountManager {
        id: accountManager
    }

    AccountSyncAdapter {
        id: accountSyncAdapter
    }

    Component {
        id: settingsLoaderComponent

        AccountSyncManager {
            signal finished

            function start(accountId) {
                if (createAllProfiles(accountId) == 0) {
                    finished()
                }
            }

            onAllProfilesCreated: finished()

            onAllProfileCreationError: {
                console.log("AccountsPage: unable to create missing profiles for account", accountId)
                finished()
            }
        }
    }

    function accountClicked(accountId, providerName, action) {
        var credentialsNeedUpdate = model.getByAccount(accountId).accountError === AccountModel.AccountNotSignedInError
        _showSettings(providerName, accountId, credentialsNeedUpdate, action)
        currentAccountId = accountId
    }

    function accountRemoveRequested(accountId) {
        accountCreationManager.deleteAccount(accountId)
    }

    function accountSyncRequested(accountId) {
        accountSyncAdapter.triggerSync(accountId)
    }
}
