/*
 * SPDX-FileCopyrightText: 2020 Jolla Ltd
 * SPDX-FileCopyrightText: 2023-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.0
import Sailfish.Silica 1.0
import Sailfish.Accounts 1.0
import Sailfish.Policy 1.0
import com.jolla.settings 1.0
import com.jolla.settings.system 1.0

SilicaFlickable {
    id: root

    property alias accountsPage: logic.accountsPage
    property alias showHeader: header.visible
    property int currentAccountId: -1

    function showCurrentPage(action) {
        if (currentAccountId >= 0) {
            const accountRef = accountsView.model.getByAccount(currentAccountId)
            if (accountRef) {
                logic.accountClicked(currentAccountId, accountRef["providerName"], action)
                return
            }
        }
        currentAccountId = -1
        logic.accountCreationManager.startAccountCreation(action)
    }

    function showPage(section, path, pageStackAction, properties) {
        if (!properties.currentAccountId) {
            return
        }

        currentAccountId = properties.currentAccountId
        if (currentAccountId > -1) {
            showCurrentPage()
        }
    }

    function startAccountCreationIfSecondPanelExists() {
        if (splitView.count !== 1) {
            logic.accountCreationManager.startAccountCreation()
        }
    }

    Component.onCompleted: {
        if (!accountsPage) {
            var parentItem = this
            while (parentItem) {
                if (parentItem.hasOwnProperty("__silica_page")) {
                    accountsPage = parentItem
                    return
                }
                parentItem = parentItem.parent
            }
        }
    }

    anchors.fill: parent
    contentHeight: content.height

    VerticalScrollDecorator {}

    AccountsViewLogic {
        id: logic
        title: header.title
        model: accountsView.model

        onRequestAccountRemoval: accountsView.removeAccount(accountId)
    }

    Column {
        id: content
        width: parent.width
        enabled: AccessPolicy.accountCreationEnabled
        spacing: SettingsUtils.isSettingsApplication ? Theme.paddingSmall : 0

        PageHeader {
            id: header
            //: Heading of the main Accounts page
            //% "Accounts"
            title: qsTrId("settings_accounts-he-page_accounts")
        }

        DisabledByMdmBanner {
            active: !AccessPolicy.accountCreationEnabled
        }

        SectionHeader {
            //: Heading of sub-list of provisioned accounts (created by MDM)
            //% "Provisioned"
            text: qsTrId("components_accounts-he-provisioned")
            visible: provisionedAccountsView.model.count > 0
        }

        AccountsFlowView {
            id: provisionedAccountsView

            filterType: AccountModel.ProvisionedFilter
            filter: "true"
            visible: provisionedAccountsView.model.count > 0
            width: parent.width
            itemWidth: SettingsUtils.isSettingsApplication ? width : (Screen.sizeCategory >= Screen.Large ? width / 2 : width)
            entriesInteractive: true

            onAccountClicked: {
                if (!SettingsUtils.isSettingsApplication || splitView.count < 2 || currentAccountId !== accountId) {
                    logic.accountClicked(accountId, providerName)
                }
            }
            onAccountRemoveRequested: logic.accountRemoveRequested(accountId)
            onAccountSyncRequested: logic.accountSyncRequested(accountId)
            onRemoveRemorseCreated: {
                logic.accountCreationManager.newCreationPageNeeded = false
                root.startAccountCreationIfSecondPanelExists()
            }
        }

        SectionHeader {
            id: personalAccountsHeader
            //: Heading of sub-list of personal accounts (created by the user)
            //% "Personal"
            text: qsTrId("components_accounts-he-personal")
            visible: provisionedAccountsView.model.count > 0
        }

        AccountsFlowView {
            id: accountsView

            filterType: AccountModel.ProvisionedFilter
            filter: "false"
            width: parent.width
            itemWidth: SettingsUtils.isSettingsApplication ? width : (Screen.sizeCategory >= Screen.Large ? width / 2 : width)
            entriesInteractive: true

            onAccountClicked: {
                if (!SettingsUtils.isSettingsApplication || splitView.count < 2 || currentAccountId !== accountId) {
                    logic.accountClicked(accountId, providerName)
                }
            }
            onAccountRemoveRequested: logic.accountRemoveRequested(accountId)
            onAccountSyncRequested: logic.accountSyncRequested(accountId)
            onRemoveRemorseCreated: {
                logic.accountCreationManager.newCreationPageNeeded = false
                root.startAccountCreationIfSecondPanelExists()
            }

            ViewPlaceholder {
                enabled: provisionedAccountsView.model.count == 0 && accountsView.model.count == 0

                //: Viewplaceholder for no accounts, no pulley menu, only add account button
                //% "No accounts"
                text: qsTrId("components_accounts-he-no_accounts_no_pulley")
            }
        }

        AddAccountButton {
            id: addItem
            
            checked: currentAccountId < 0 && SettingsUtils.isMultiPanelMode()
            _settingsApplicationStyling: SettingsUtils.isSettingsApplication

            onClicked: {
                if (!SettingsUtils.isSettingsApplication || splitView.count < 2 || currentAccountId != -1) {
                    logic.accountCreationManager.startAccountCreation()
                }
            }
        }
    }
}
