/****************************************************************************************
**
** SPDX-FileCopyrightText: Copyright 2023-2024 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: Proprietary
**
****************************************************************************************/

import QtQuick 2.0
import Sailfish.Silica 1.0
import Sailfish.Accounts 1.0
import org.nemomobile.systemsettings 1.0

Item {
    id: root

    property Account account: accountManager.account(accountId)
    property Provider accountProvider
    property bool accountEnabled
    property bool accountEnabledReadOnly
    property bool accountIsProvisioned
    property string accountUserName
    property alias accountDisplayName: accountDisplayNameField.text
    property var nextFocusItem

    AboutSettings {
        id: aboutSettings
    }

    width: parent.width
    height: statusCombo.height
            + statusCombo.anchors.topMargin
            + accountDisplayNameField.height
            + Theme.paddingLarge

    onAccountEnabledChanged: statusCombo.currentIndex = accountEnabled ? 0 : 1
    
    ComboBox {
        id: statusCombo
        
        enabled: !root.accountEnabledReadOnly
        description: {
            if (root.accountIsProvisioned && root.accountEnabledReadOnly) {
                if (accountEnabled) {
                    //: Indicates that the account is currently enabled by MDM
                    //: %1 is an operating system name without the OS suffix
                    //% "Enabled by %1 Device Manager"
                    return qsTrId("settings-accounts-la-account_enabled_by_mdm")
                        .arg(aboutSettings.baseOperatingSystemName)
                } else {
                    //: Indicates that the account is currently disabled by MDM
                    //: %1 is an operating system name without the OS suffix
                    //% "Disabled by %1 Device Manager"
                    return qsTrId("settings-accounts-la-account_disabled_by_mdm")
                        .arg(aboutSettings.baseOperatingSystemName)
                }
            } else {
                return ""
            }
        }

        //: Indicates whether the account is currently enabled
        //% "Account status"
        label: qsTrId("settings-accounts-me-account_status")

        menu: ContextMenu {
            MenuItem {
                //: Indicates the account is currently enabled and active
                //% "Active"
                text: qsTrId("settings-accounts-me-account_active")

                onClicked: model.setAccountEnabled(accountId, true)
            }

            MenuItem {
                //: Indicates the account is currently not enabled
                //% "Disabled"
                text: qsTrId("settings-accounts-me-account_disabled")

                onClicked: model.setAccountEnabled(accountId, false)
            }
        }
    }

    TextField {
        id: accountDisplayNameField

        anchors.top: statusCombo.bottom

        //: Short name or summary for a user account
        //% "Description"
        label: qsTrId("components_accounts-la-account_description")

        EnterKey.iconSource: !!root.nextFocusItem
                             ? "image://theme/icon-m-enter-next"
                             : "image://theme/icon-m-enter-close"
                             
        EnterKey.onClicked: {
            if (!!root.nextFocusItem) {
                root.nextFocusItem.focus = true
            } else {
                root.focus = true
            }
        }
    }

    Connections {
        target: account

        onEnabledChanged: accountEnabled = account.enabled
    }

    Component.onCompleted: statusCombo.currentIndex = accountEnabled ? 0 : 1
}
