/*
 * SPDX-FileCopyrightText: 2023-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.6
import Sailfish.Silica 1.0
import Sailfish.Silica.private 1.0
import Sailfish.Lipstick 1.0
import com.jolla.settings 1.0

SilicaMouseArea {
    id: root

    property real controlHeight: parent && parent.hasOwnProperty('mediator') ? parent.mediator.controlHeight : 0.0
    property real maximumValue: 1.0
    property real minimumValue
    readonly property real _range: maximumValue - minimumValue
    property real stepSize
    property real value
    readonly property real sliderValue: Math.max(minimumValue, Math.min(maximumValue, value))

    property alias label: labelText.text
    readonly property bool down: pressed && !DragFilter.canceled

    property alias iconSource: highlight.source
    property real controlScale: 1.0
    property real labelScale: 1.0
    property bool labelVisible: true

    property bool _animateValue: calculateAnimateValue()

    property real _precFactor: 1.0

    readonly property real _grooveWidth: Math.max(0, background.width - progressBar.height - progressBar.anchors.rightMargin - progressBar.anchors.leftMargin)

    // ascent enough to keep text in slider area
    readonly property int _backgroundTopPadding: (Theme.itemSizeMedium - background.height) * 0.5 - Theme.paddingSmall * 0.5

    // Animation should be disabled when controlScale value changes from 1.0 to 0.8 and vice versa.
    // We get something like 0.800000011920929 at the end of the animation. This is why we have to use rounding.
    function calculateAnimateValue() {
        return controlScale === 1.0 || Math.round(controlScale * 100) === 80
    }

    function _updateValueToDraggable() {
        var pos = draggable.x - progressBar.height
        value = _calcValue((pos / _grooveWidth) * _range + minimumValue)
    }

    function _calcValue(newVal) {
        if (newVal <= minimumValue) {
            return minimumValue
        }

        if (stepSize > 0.0) {
            var offset = newVal - minimumValue
            var intervals = Math.round(offset / stepSize)
            newVal = Math.round((minimumValue + (intervals * stepSize)) * _precFactor) / _precFactor
        }

        if (newVal > maximumValue) {
            return maximumValue
        }

        return newVal
    }

    onPreventStealingChanged: if (preventStealing) root.DragFilter.end()

    onStepSizeChanged: {
        // Avoid rounding errors.  We assume that the range will
        // be sensibly related to stepSize
        var decimial = Math.floor(stepSize) - stepSize
        if (decimial < 0.001) {
            _precFactor = Math.pow(10, 7)
        } else if (decimial < 0.1) {
            _precFactor = Math.pow(10, 4)
        } else {
            _precFactor = 1.0
        }
    }

    onPressed: {
        _animateValue = false
        root.DragFilter.begin(mouse.x, mouse.y)

        draggable.x = Math.min(Math.max(drag.minimumX, mouseX + progressBar.height * 0.5), drag.maximumX)
    }

    onReleased: {
        _animateValue = Qt.binding(calculateAnimateValue)
        _updateValueToDraggable()
    }

    onCanceled: {
        root.DragFilter.end()
    }

    width: parent.width - Theme.dp(23) * 2

    DragFilter.orientations: Qt.Vertical

    implicitHeight: _backgroundTopPadding + background.height * 0.5 + 2 * Theme.paddingMedium + labelText.implicitHeight

    drag {
        onActiveChanged: if (drag.active && !root.DragFilter.canceled) root.DragFilter.end()
        target: draggable
        minimumX: progressBar.height
        maximumX: background.width - progressBar.anchors.leftMargin - progressBar.anchors.rightMargin
        axis: Drag.XAxis
    }

    Item {
        id: backgroundContainer

        anchors.horizontalCenter: parent.horizontalCenter

        width: root.width
        height: root.controlHeight
    }

    Rectangle {
        id: background

        anchors.fill: backgroundContainer
        anchors.margins: (root.controlHeight - root.controlScale * root.controlHeight) * 0.5

        radius: Math.floor(height * 0.5)
        color: Theme.rgba(Theme.primaryColor, Theme.opacityFaint)
    }

    Rectangle {
        id: progressBar

        anchors {
            top: background.top
            left: background.left
            bottom: background.bottom
            margins: Theme.paddingSmall
        }

        width: ((sliderValue - minimumValue) * _grooveWidth / _range) + progressBar.height

        radius: Math.floor(height * 0.5)
        color: Qt.tint(Theme.highlightDimmerColor, Theme.rgba(Theme.highlightColor, root.down ? 0.7 : 0.9))

        Behavior on width {
            enabled: _animateValue
            SmoothedAnimation {
                duration: 200
                easing.type: Easing.OutQuad
            }
        }
    }

    HighlightImage {
        id: highlight

        anchors {
            right: progressBar.right
            rightMargin: (progressBar.height - highlight.height) * 0.5
            verticalCenter: background.verticalCenter
        }

        height: Theme.iconSizeSmall
        width: height
        scale: root.controlScale
        color: Theme.colorScheme === Theme.LightOnDark ? Theme.darkPrimaryColor : Theme.lightPrimaryColor
    }

    Item {
        id: draggable

        onXChanged: {
            if (root.DragFilter.canceled) {
                return
            }

            if (root.drag.active) {
                _updateValueToDraggable()
            }
        }
    }

    CaptionLabel {
        id: labelText

        anchors {
            horizontalCenter: parent.horizontalCenter
            top: backgroundContainer.bottom
            topMargin: Theme.paddingSmall * 0.5
        }
        width: parent.width - Theme.dp(4)
        visible: root.labelVisible && text.length
        scale: root.labelScale
    }
}
