/**
* SPDX-FileCopyrightText: 2023-2025 Open Mobile Platform LLC <community@omp.ru>
* SPDX-License-Identifier: Proprietary
*/

import QtQuick 2.0
import Sailfish.Silica 1.0
import Nemo.DBus 2.0
import com.jolla.settings 1.0 // Load translations
import Sailfish.Lipstick 1.0
import org.nemomobile.lipstick 0.1
import ru.omp.authclient 1.0

SettingsControl {
    id: root

    // Whether the switch is "available" for changing state. Can be unavailable e.g. due to flight mode.
    // Mouse events are passed through.
    property bool available: true

    // Whether the switch is in the "checked" state.
    property bool checked

    // Whether the setting is "active". While "checked" indicates when a setting is "on", "active"
    // indicates some additional state; for example, when a connection has been established.
    property bool active

    // Whether the switch is in a "busy" state and should not be enabled.
    property bool busy

    // Whether the switch shows on/off label on click
    property bool showOnOffLabel: true

    // The text to be displayed when active=true.
    property string activeText: name

    property string popupMenuTitle: label.text

    // The setting icon image.
    property alias icon: iconImage

    property string systemIcon: entryParams["system_icon"] || "image://theme/icon-system-resources"

    property alias errorNotification: errorNotification

    property bool _pendingActionAfterUnlock: false

    property bool stateToggle 

    readonly property color _borderColor: Theme.rgba(down ? Theme.highlightColor : Theme.primaryColor, 0.1)
    readonly property real _borderWidth: available || checked ? 0 : Theme.dp(2)
    property color _backgroundColor: {
        if (!available && !checked) {
            return Theme.rgba(down ? Theme.highlightColor : Theme.primaryColor, 0.05)
        }

        if (checked) {
            var tintAlpha = (available ? 0.9 : 0.5) - (down ? 0.2 : 0.0)
            return Qt.tint(Theme.highlightDimmerColor, Theme.rgba(Theme.highlightColor, tintAlpha))
        }

        return down ? Theme.rgba(Theme.highlightColor, 0.5) : Theme.rgba(Theme.primaryColor, Theme.opacityFaint)
    }

    property color _controlColor: {
        if (down) {
            return Theme.rgba(Theme.highlightColor, _controlOpacity)
        }

        if (checked) {
            return Theme.rgba(Theme.colorScheme === Theme.LightOnDark
                              ? Theme.darkPrimaryColor
                              : Theme.lightPrimaryColor,
                              _controlOpacity)
        }

        return Theme.rgba(Theme.primaryColor, _controlOpacity)
    }

    property real _controlOpacity: {
        if (!available) {
            return checked ? 0.6 : Theme.opacityLow
        }
        return 1
    }

    readonly property bool _doubleControl: entryParams.columns === 2

    default property alias controlContent: contentItem.data
    property int __jolla_settings_toggle

    // Emitted when the "checked" state should be toggled.
    // Note: Use this instead of onClicked, to ensure user access restrictions are correctly filtered.
    signal toggled()

    signal pendingActionAfterUnlockTriggered()

    function callActionAfterUnlock() {
        _pendingActionAfterUnlock = false
        if (AuthDevlockClient.state === AuthDevlockClient.Locked) {
            _pendingActionAfterUnlock = true
            root.mediator.showUnlockScreen()
        } else {
            pendingActionAfterUnlockTriggered()
        }
    }

    function showPopupMenu() {
        circle.showPopup()
    }

    width: implicitWidth
    contentHeight: Theme.iconSizeLauncher + Theme.paddingSmall

    settingsPageEntryPath: entryParams["settings_path"] || ""
    _showPress: false
    openMenuOnPressAndHold: false

    acceptedButtons: Qt.LeftButton | Qt.RightButton

    onPressAndHold: circle.showPopup()

    onClicked: {
        if (mouse.button === Qt.RightButton) {
            pressAndHold(mouse)
            return
        }

        if (busy) {
            errorNotification.notify(SettingsControlError.InProgress)
            return
        }

        stateToggle = !checked
        if (userAccessRestricted) {
            requestUserAccess()
        } else {
            toggled()
        }
    }

    PopupPageItem {
        id: circle

        anchors {
            fill: controlArea
            margins: root._doubleControl ? (controlArea.height - root._controlScale * controlArea.height) * 0.5 : 0.0
        }

        contentComponent: root.popupMenu
        label: root.popupMenuTitle
        highlightRadius: Math.floor(height * 0.5)
        highlightOpacity: 1.0
        highlightColor: _backgroundColor
        scale: root._doubleControl ? 1.0 : root._controlScale
        highlightBorder.width: _borderWidth
        highlightBorder.color: _borderColor

        HighlightImage {
            id: iconImage

            anchors {
                left: parent.left
                leftMargin: (parent.height - height) * 0.5
                verticalCenter: parent.verticalCenter
            }
            visible: iconImage.status === Image.Ready
            scale: root._doubleControl ? root._controlScale : 1.0

            // NOTE: We cannot use HighlightImage.color to control opacity of white
            // icon in the LightOnDark theme.
            opacity: root._controlOpacity
            color: Theme.rgba(root._controlColor, 1.0)
        }

        Item {
            id: contentItem

            anchors.right: parent.right
            anchors.left: iconImage.status === Image.Ready ? iconImage.right : parent.left
            anchors.leftMargin: iconImage.status === Image.Ready
                                ? Theme.paddingSmall - (iconImage.width - iconImage.width * root._controlScale) * 0.5
                                : 0
            height: parent.height
        }
    }

    CaptionLabel {
        id: label

        anchors {
            horizontalCenter: parent.horizontalCenter
            top: controlArea.bottom
        }
        width: root.width - Theme.dp(4)
        visible: root._labelVisible
        scale: root._labelScale
        text: (root.active && root.activeText.length > 0) ? root.activeText : root.name
    }

    Item {
        id: controlArea

        anchors.horizontalCenter: parent.horizontalCenter

        width: root._doubleControl ? root.mediator.doubleControlWidth : height
        height: root.mediator.controlHeight
    }

    SettingsErrorNotification {
        id: errorNotification
    }

    Connections {
        target: AuthDevlockClient

        onStateChanged: {
            if (_pendingActionAfterUnlock && AuthDevlockClient.state === AuthDevlockClient.Unlocked) {
                pendingActionAfterUnlockTriggered()
            }
            _pendingActionAfterUnlock = false
        }
    }

    Connections {
        target: root.mediator

        onLockCodeEntryClosed: {
            if (AuthDevlockClient.state !== AuthDevlockClient.Unlocked) {
                // User moved away from lock code entry screen without unlocking the device,
                // so cancel the pending triggering of the action.
                _pendingActionAfterUnlock = false
            }
        }
    }
}
