import QtQuick 2.0
import Sailfish.Silica 1.0
import Sailfish.Timezone 1.0

Page {
    id: page
    property bool showNoTimezoneOption
    property bool showUniversalTimeOption

    signal timezoneClicked(string name)

    SilicaListView {
        id: view

        anchors.fill: parent
        currentIndex: -1
        focus: true
        model: TimezoneModel { }
        header: Column {
            width: view.width

            // When the extra "No time zone" and "UTC" options are shown/hidden, the header size
            // changes and causes the contentY to jump, so adjust the contentY for this change.
            property real _prevHeight: height
            onHeightChanged: {
                var delta = (height - _prevHeight)
                _prevHeight = height
                if (delta > 0) {
                    view.contentY -= delta
                }
            }

            PageHeader {
                id: pageHeader

                //% "Time zone"
                title: qsTrId("components_timezone-he-time_zone")
            }

            Item {
                id: searchFieldPlaceholder
                width: parent.width
                height: page.isLandscape ? 0 : searchField.height
            }

            SearchField {
                id: searchField

                parent: page.isLandscape ? pageHeader.extraContent : searchFieldPlaceholder

                width: parent.width
                anchors.verticalCenter: parent.verticalCenter

                focus: true
                EnterKey.iconSource: "image://theme/icon-m-enter-close"
                EnterKey.onClicked: focus = false

                Binding {
                    target: view.model
                    property: "filter"
                    value: searchField.text.toLowerCase().trim()
                }
            }

            Connections {
                target: searchField.activeFocus ? view : null
                ignoreUnknownSignals: true
                onContentYChanged: {
                    if (view.contentY > (Screen.height / 2)) {
                        searchField.focus = false
                    }
                }
            }

            BackgroundItem {
                visible: page.showNoTimezoneOption && (searchField.text.length == 0)
                height: Theme.itemSizeSmall
                Label {
                    x: Theme.horizontalPageMargin
                    anchors.verticalCenter: parent.verticalCenter
                    //% "No time zone"
                    text: qsTrId("components_timezone-la-no_time_zone")
                }
                onClicked: page.timezoneClicked("")
            }

            BackgroundItem {
                visible: page.showUniversalTimeOption && (searchField.text.length == 0)
                height: Theme.itemSizeSmall
                Label {
                    anchors {
                        verticalCenter: parent.verticalCenter
                        left: parent.left
                        leftMargin: Theme.horizontalPageMargin
                        right: parent.right
                        rightMargin: Theme.horizontalPageMargin
                    }
                    //% "Coordinated universal time (UTC)"
                    text: qsTrId("components_timezone-la-utc")
                    wrapMode: Text.Wrap
                }
                onClicked: page.timezoneClicked("UTC")
            }
        }
        delegate: BackgroundItem {
            id: background
            property bool highlight: background.down || view.model.timezone == model.name
            height: Theme.itemSizeMedium
            onClicked: page.timezoneClicked(model.name)
            Label {
                id: countryLabel

                anchors {
                    verticalCenter: parent.verticalCenter
                    verticalCenterOffset: -subLabel.implicitHeight / 2
                    left: parent.left
                    leftMargin: Theme.horizontalPageMargin
                    right: parent.right
                    rightMargin: Theme.horizontalPageMargin
                }
                textFormat: Text.StyledText
                text: Theme.highlightText(model.country, view.model.filter, Theme.highlightColor)
                color: background.highlight ? Theme.highlightColor : Theme.primaryColor
                wrapMode: Text.Wrap
            }
            Label {
                id: subLabel
                x: Theme.horizontalPageMargin
                anchors.top: countryLabel.bottom
                textFormat: Text.StyledText
                text: model.offsetWithDstOffset + " " + Theme.highlightText(model.city, view.model.filter, Theme.highlightColor)
                font.pixelSize: Theme.fontSizeSmall
                color: background.highlight ? Theme.highlightColor : Theme.secondaryColor
            }
        }
        section.property: "sectionOffset"
        section.delegate: SectionHeader {
            height: Theme.itemSizeExtraSmall
            text: section
        }

        VerticalScrollDecorator {}
    }
}
