/****************************************************************************
**
** SPDX-FileCopyrightText: 2022-2025 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: Proprietary
**
****************************************************************************/

import QtQuick 2.0
import Nemo.DBus 2.0
import Sailfish.Silica 1.0
import Aurora.Controls 1.0
import ru.omp.amberpdf 1.0
import ru.omp.print 1.0

Dialog {
    id: root

    readonly property alias colorModes: printer.supportedColorModes
    readonly property alias duplexModes: printer.supportedDuplexModes
    readonly property alias pageSizes: printer.supportedPageSizes
    readonly property alias orientations: printer.supportedOrientations
    property alias printerSettings: printer.settings

    property var loadedPrinters: printerManager.savedPrinters
    property int numberOfCopies: 1

    property string pageRange
    property string printPath
    property bool centerPagesInLandscapeOrientation: !imgToPdfConverter.isActivated
    property bool _printersLoaded
    property bool _printerSettingsLoaded
    property bool _readyForPreview
    readonly property bool _conversionSuccessful: {
        if (!_readyForPreview || !printPath) {
            return false
        }
        
        const fileName = printPath.split("/").pop()
        const fileParts = fileName.split(".")
        
        if (fileParts.length <= 1) {
            return false
        }
        
        const fileExtension = fileParts.pop().toLowerCase()
        return (fileExtension === "pdf")
    }

    readonly property string _pdfPrinter: "PDF-ccf7-4753-bc3c-d83c86c1a9cc"
    readonly property bool _isPDFPrinter: printer.name === _pdfPrinter

    function toastShow(text) {
        Notices.show(
                    text,
                    Notice.Short,
                    Notice.Bottom,
                    0,
                    -Theme.horizontalPageMargin)
    }

    function updatePageRange(updatedPageRange) {
        pageSelection.visible = true
        pageSelection.text = updatedPageRange
        pageRange = pageSelection.text
    }

    function launchToPdfConvert() {
        _readyForPreview = false

        if (!printerSettings.format) {
            //% "Failed to receive paper format"
            toastShow(qsTrId("components_pickers-la-failed-to-receive-paper-format"))
        } else {
            imgToPdfConverter.launchToConvert(
                        printerSettings.format.width,
                        printerSettings.format.height,
                        printerSettings.orientation == PrinterSettings.Portrait,
                        /*scaleToPaper=*/true)
        }
    }

    function updateLoadedPrinters(printers) {
        loadedPrinters = printers
        if (loadedPrinters.length == 0) {
            return
        }

        var printerJsonString = JSON.parse(loadedPrinters[0])
        printer.name = printerJsonString.name
        _printersLoaded = true
    }

    onStatusChanged: {
        if (page.status === PageStatus.Active && !acceptDestination) {
            acceptDestination = pageStack.previousPage(page)
        }
    }

    onAccepted: {
        printer.print(printPath, pageRange, numberOfCopies)

        //% "The document was successfully sent for printing"
        toastShow(qsTrId("components_pickers-la-document-was-successfully-sent-for-printing"))
    }

    on_ConversionSuccessfulChanged: {
        if (!_conversionSuccessful) {
            //% "File conversion to PDF failed. Format may not be supported."
            toastShow(qsTrId("components_pickers-la-conversion-failed"))
        }
    }

    onRejected: printer.requestTmpFileRemoval(printPath)

    canAccept: loadedPrinters.length !== 0
                && (!pageSelection.visible || pageSelection.acceptableInput)
                && (!_isPDFPrinter || printerSettings.printDirectory.length !== 0) && _conversionSuccessful
    acceptDestinationAction: PageStackAction.Pop

    Printer {
        id: printer

        onNameChanged: {
            printerSettings.format = null
            printerSettings.duplexMode = twoSideSwitch.checked ? PrinterSettings.DuplexAuto
                                                               : PrinterSettings.DuplexNone
            printerSettings.colorMode = colorModeSwitch.checked ? PrinterSettings.Color
                                                                : PrinterSettings.GrayScale
            //% "Default"
            pickPageSizeButton.value = qsTrId("components_pickers-la-paper-default")
            _readyForPreview = false
        }

        onCompleteUpdateSupportedSettings: {
            var hasA4 = false
            for (var index = 0; index < supportedPageSizes.length; ++index) {
                const prop = supportedPageSizes[index]
                if (prop.name == "A4") {
                    printerSettings.format = prop
                    pickPageSizeButton.value = prop.name
                    hasA4 = true
                    break;
                }
            }

            if (!hasA4 && supportedPageSizes.length > 0) {
                const prop = supportedPageSizes[0]
                printerSettings.format = prop
                pickPageSizeButton.value = prop.name
            }
            launchToPdfConvert()

            printerSettings.outFileName = fileNameTextField.text
            _printerSettingsLoaded = true
        }
    }

    PrinterManager {
        id: printerManager

        onSavedPrintersChanged: updateLoadedPrinters(printers)
        onSavedPrintersLoaded: updateLoadedPrinters(printers)
    }

    PdfDocument {
        id: pdfDocument

        onStatusChanged: {
            if (status == PdfDocument.Ready) {
                pageSelection.validator.pagesCount = count()
            }
        }
    }

    ImgToPdfConverter {
        id: imgToPdfConverter

        onOutputFilePathChanged: {
            root.printPath = imgToPdfConverter.outputFilePath
        }

        onStateChanged: {
            if (state == ImgToPdfConverter.Finished) {
                _readyForPreview = true
                pdfDocument.path = root.printPath

                if (!root._conversionSuccessful) {
                    //% "File conversion to PDF failed. Format may not be supported."
                    toastShow(qsTrId("components_pickers-la-conversion-failed"))
                }
            } else {
                _readyForPreview = false
            }
        }
    }

    Component {
        id: pdfFolderPicker

        FolderPickerPage {
            onSelectedPathChanged: {
                if (printerSettings.printDirectory != selectedPath) {
                    printerSettings.printDirectory = selectedPath
                }
            }

            showSystemFiles: false

            //% "Select a directory"
            dialogTitle: qsTrId("settings_printers-la-select-print-directory")

            selectedPath: printerSettings.printDirectory
        }
    }

    Timer {
        id: timer

        onTriggered: _printersLoaded = true
        interval: 5000
    }

    DBusInterface {
        id: settingsDBus

        service: "com.jolla.settings"
        path: "/com/jolla/settings/ui"
        iface: "com.jolla.settings.ui"
    }

    SilicaFlickable {
        width: parent.width
        height: parent.height - Theme.paddingLarge * 2 - toolBar.height
        contentHeight: column.height + toolBar.height
        anchors.top: printHeader.bottom

        DialogHeader {
            id: printHeader

            //% "Print"
            acceptText: qsTrId("components_pickers-he-print")
        }

        BusyIndicator {
            id: busyIndicator

            anchors {
                top: printHeader.bottom
                topMargin: Theme.paddingLarge
                horizontalCenter: parent.horizontalCenter
            }

            running: !_printersLoaded
            size: BusyIndicatorSize.Large
        }

        Label {
            anchors {
                top: busyIndicator.bottom
                topMargin: Theme.paddingLarge
                horizontalCenter: parent.horizontalCenter
            }
            highlighted: true
            opacity: Theme.opacityHigh

            //% "Loading saved printer"
            text: qsTrId("settings_printers-la-loading_saved_printers")
            visible: !_printersLoaded
        }

        BusyIndicator {
            id: settingsBusyIndicator

            anchors {
                top: printHeader.bottom
                topMargin: Theme.paddingLarge + printerSelector.height + printingSettingsHeader.height
                           + copiesTextField.height + (Theme.paddingMedium - 2 * column.spacing)
                           + pageRangeComboxBox.height + pageSelection.height + 5 * column.spacing
                horizontalCenter: parent.horizontalCenter
            }

            running: canAccept && !_printerSettingsLoaded
            size: BusyIndicatorSize.Large
        }

        Label {
            anchors {
                top: settingsBusyIndicator.bottom
                topMargin: Theme.paddingLarge
                horizontalCenter: parent.horizontalCenter
            }
            highlighted: true
            opacity: Theme.opacityHigh

            //% "Loading printer settings"
            text: qsTrId("settings_printers-la-loading_printer_settings")
            visible: canAccept && !_printerSettingsLoaded
        }

        Column {
            id: column

            anchors {
                top: printHeader.bottom
                topMargin: Theme.paddingMedium
            }
            width: parent.width
            spacing: Theme.paddingSmall
            visible: loadedPrinters.count !== 0

            ViewPlaceholder {
                enabled: loadedPrinters.length === 0 && _printersLoaded
                //% "No added printers found"
                text: qsTrId("components_pickers-he-no_added_printers")
                //% "Go to settings to add printer"
                hintText: qsTrId("components_pickers-he-extended_settings")
            }

            ComboBox {
                id: printerSelector

                currentIndex: 0
                //% "Printer"
                label: qsTrId("components_pickers-la-printer")

                //% "Print to pdf"
                value: _isPDFPrinter ? qsTrId("components_pickers-he-print_to_pdf") : printer.name
                visible: loadedPrinters.length !== 0

                menu: ContextMenu {
                    Repeater {
                        model: loadedPrinters
                        delegate: MenuItem {
                            onClicked: {
                                printer.name = (text == qsTrId("components_pickers-he-print_to_pdf")
                                               ? JSON.parse(modelData).name
                                               : text)
                            }

                            text: JSON.parse(modelData).name == _pdfPrinter
                                                //% "Print to pdf"
                                                ? qsTrId("components_pickers-he-print_to_pdf")
                                                : JSON.parse(modelData).name
                        }
                    }

                    MenuItem {
                        onClicked: settingsDBus.call("showAddPrinterDialog")

                        //% "Add new printer"
                        text: qsTrId("components_pickers-la-add_new_printer")
                    }
                }
            }

            Item {width: 1; height: Theme.paddingMedium - 2 * column.spacing}

            TextField {
                id: fileNameTextField

                onTextChanged: printerSettings.outFileName = text

                text: (printPath.split("/").pop()).split(".")[0]

                //% "File name"
                label: qsTrId("libprinter-qml-plugin-la-file_name")

                visible: loadedPrinters.length !== 0 && _isPDFPrinter
                errorHighlight: text.length == 0
            }

            ComboBox {
                onClicked: pageStack.animatorPush(pdfFolderPicker)

                //% "Folder"
                label: qsTrId("settings_printers-la-folder")
                visible: loadedPrinters.length !== 0 && _isPDFPrinter

                value: printerSettings.printDirectory.split("/").pop()
            }

            SectionHeader {
                id: printingSettingsHeader

                //% "Printing settings"
                text: qsTrId("components_pickers-la-printing-settings")
                horizontalAlignment: Text.AlignLeft
                visible: loadedPrinters.length !== 0
            }

            TextField {
                id: copiesTextField

                onTextChanged: numberOfCopies = parseInt(text)

                text: "1"
                //% "Number of copies"
                label: qsTrId("libprinter-qml-plugin-la-number-of-copies")
                inputMethodHints: Qt.ImhFormattedNumbersOnly
                validator: IntValidator { bottom: 1 }
                visible: loadedPrinters.length !== 0 && !_isPDFPrinter
            }

            ComboBox {
                id: pageRangeComboxBox

                //% "Page range"
                label: qsTrId("components_pickers-la-page-range")
                                            //% "Select Pages"
                value: pageSelection.visible ? qsTrId("components_pickers-la-select-pages")
                                            //% "All"
                                            : qsTrId("components_pickers-la-all")
                visible: loadedPrinters.length !== 0 && pdfDocument.status === PdfDocument.Ready

                menu: ContextMenu {
                    MenuItem {
                        onClicked: {
                            if (pageSelection.visible) {
                                pageSelection.visible = false
                            }
                            pageRange = ""
                        }

                        //% "All"
                        text: qsTrId("components_pickers-la-all")
                    }

                    MenuItem {
                        onClicked: {
                            pageSelection.visible = true
                            pageSelection.forceActiveFocus()
                        }

                        //% "Select Pages"
                        text: qsTrId("components_pickers-la-select-pages")
                    }
                }
            }

            TextField {
                id: pageSelection

                onTextChanged: pageRange = text
                visible: false
                //% "Specify the range of pages (for example 1-5) or list the pages required for printing separated by commas (for example 2, 6, 18)"
                label: qsTrId("printer-qml-plugin-la-page-selection-description")
                inputMethodHints: Qt.ImhFormattedNumbersOnly
                validator: PageRangeValidator{}
            }

            Column {
                width: parent.width
                spacing: Theme.paddingSmall
                visible: _printerSettingsLoaded

                ComboBox {
                    id: pageOrderComboBox

                    //% "Page order"
                    label: qsTrId("components_pickers-la-page-order")

                    value: printerSettings.pageOrder == PrinterSettings.Direct
                                //% "Direct"
                                ? qsTrId("components_pickers-la-direct_order")
                                //% "Reverse"
                                : qsTrId("components_pickers-la-reverse_order")
                    visible: loadedPrinters.length !== 0

                    menu: ContextMenu {
                        MenuItem {
                            onClicked: printerSettings.pageOrder = PrinterSettings.Direct

                            //% "Direct"
                            text: qsTrId("components_pickers-la-direct_order")
                        }

                        MenuItem {
                            onClicked: printerSettings.pageOrder = PrinterSettings.Reverse

                            //% "Reverse"
                            text: qsTrId("components_pickers-la-reverse_order")
                        }
                    }
                }

                ValueButton {
                    id: pickPageSizeButton

                    onClicked: {
                        if (pageSizes.length > 0) {
                            const dialog = pageStack.push("PageSizePickerDialog.qml",
                                {
                                    formats: root.pageSizes,
                                    currentFormat: printerSettings.format ? printerSettings.format.name : ""
                                })

                            dialog.formatClicked.connect(function(format) {
                                if (format) {
                                    value = format.name
                                }
                                printerSettings.format = format
                                if (imgToPdfConverter.isActivated) {
                                    launchToPdfConvert()
                                }
                            })
                        } else {
                            //% "No info about supported page size found"
                            toastShow(qsTrId("components_pickers-la-no_page_size"))
                        }
                    }
                    //% "Paper size"
                    label: qsTrId("components_pickers-la-paper-size")
                    value: printerSettings.format ? printerSettings.format.name
                                                  //% "Default"
                                                  : qsTrId("components_pickers-la-paper-default")
                    visible: loadedPrinters.length !== 0
                }

                ComboBox {
                    id: orientationComboBox

                    //% "Orientation"
                    label: qsTrId("components_pickers-la-orientation")
                    value: printerSettings.orientation == PrinterSettings.Portrait
                                //% "Portrait"
                                ? qsTrId("components_pickers-la-portrait")
                                //% "Landscape"
                                : qsTrId("components_pickers-la-landscape")
                    visible: loadedPrinters.length !== 0

                    menu: ContextMenu {
                        MenuItem {
                            onClicked: {
                                printerSettings.orientation = PrinterSettings.Portrait
                                if (imgToPdfConverter.isActivated) {
                                    launchToPdfConvert()
                                }
                            }

                            //% "Portrait"
                            text: qsTrId("components_pickers-la-portrait")
                        }

                        MenuItem {
                            onClicked: {
                                printerSettings.orientation = PrinterSettings.Landscape
                                if (imgToPdfConverter.isActivated) {
                                    launchToPdfConvert()
                                }
                            }

                            //% "Landscape"
                            text: qsTrId("components_pickers-la-landscape")
                        }
                    }
                }


                ComboBox {
                    onClicked: {
                        if (!colorModeSwitch.enabled) {
                            //% "Option not supported by printer"
                            toastShow(qsTrId("components_pickers-la-option_not_supported"))
                        }
                    }

                    visible: loadedPrinters.length !== 0 && !_isPDFPrinter

                    TextSwitch {
                        id: colorModeSwitch

                        onClicked: printerSettings.colorMode = checked ? PrinterSettings.Color
                                                                       : PrinterSettings.GrayScale
                        //% "Color print"
                        text: qsTrId("components_pickers-la-color-print")
                        enabled: colorModes.length > 1
                        checked: colorModes.indexOf(PrinterSettings.Color)
                    }
                }

                ComboBox {
                    onClicked: {
                        if (!twoSideSwitch.enabled) {
                            //% "Option not supported by printer"
                            toastShow(qsTrId("components_pickers-la-option_not_supported"))
                        }
                    }

                    visible: loadedPrinters.length !== 0 && !_isPDFPrinter

                    TextSwitch {
                        id: twoSideSwitch

                        onClicked: printerSettings.duplexMode = checked ? PrinterSettings.DuplexAuto
                                                                        : PrinterSettings.DuplexNone
                        //% "Two-side print"
                        text: qsTrId("components_pickers-la-two-side-print")
                        enabled: duplexModes.length > 1
                    }
                }

                Item {
                    width: 1
                    height: Theme.paddingLarge
                }
            }
        }

        VerticalScrollDecorator {}
    }

    AppBar {
        id: toolBar

        _bottomPosition: true

        background: true
        divider: false
        visible: canAccept && !pageSelection.activeFocus

        AppBarSpacer {}

        AppBarButton {
            id: componentButton
            enabled: _readyForPreview && _conversionSuccessful 

            onClicked: {
                if (pageSelection.activeFocus) {
                    pageSelection.activeFocus = false
                }

                var objPreview = pageStack.push("PrintingPreview.qml", {
                                     "printPath": printPath,
                                     "currentPageRange": pageRange,
                                     "centerPagesInLandscapeOrientation": centerPagesInLandscapeOrientation,
                                     "printerSettings": printerSettings,
                                     "acceptDestination": root.acceptDestination,
                                     "acceptDestinationAction": root.acceptDestinationAction})
                objPreview.updatePageRange.connect(updatePageRange)
                objPreview.accepted.connect(accepted)
            }
            //% "Preview"
            text: qsTrId("components_pickers-bt-preview")

            icon.source: "image://theme/icon-m-file-other"
        }

        AppBarSpacer {}
    }

    Component.onCompleted: {
        timer.start()
        imgToPdfConverter.inputFilePath = printPath
        printerSettings.outFileName = (printPath.split("/").pop()).split(".")[0]
        printerManager.loadSavedPrinters()
    }
}
