/****************************************************************************
 **
 ** SPDX-FileCopyrightText: 2023-2024 Open Mobile Platform LLC <community@omp.ru>
 ** SPDX-License-Identifier: Proprietary
 **
 ****************************************************************************/

import QtQuick 2.0
import Sailfish.Silica 1.0

SilicaControl {
    id: item

    readonly property Image image: !!sourceImageLoader.item && sourceImageLoader.item.status === Image.Ready
                                   ? sourceImageLoader.item : defaultImage
    readonly property var status: image.status // Image compatibility

    property string icon
    property bool pressed
    property bool drawIcon: true
    property bool scaleSource: iconMode === "Scale" // by default all icons should be cropped.
    // IconMode isn't set for X-Aurora-Application prior to 5.0, so we're falling back to default
    // behavior: icons drawing as is without scaling but with cropping by system mask.
    property string iconMode: launcherItem
                              ? launcherItem.readValue("X-Aurora-Application", "IconMode")
                              : ""
    property real size: Theme.iconSizeLauncher
    property alias shadow: shader.dropShadow
    property alias highlightColor: shader.highlightColor
    property alias fillColor: shader.fillColor
    property alias outlineColor: shader.outlineColor
    property alias outlineWidth: shader.outlineWidth
    property alias shadowColor: shader.shadowColor

    // Here we can use two APIs to determine IconMode.
    // Icon from LauncherItem has the highest priority,
    // and the IconMode is automatically determined from the Desktop-file.
    // Where it is impossible to use LauncherItem, we cat use "icon"
    // and "scaleSource" properties to manage IconMode.
    property var launcherItem
    readonly property string _icon: launcherItem ? launcherItem.iconId : item.icon

    width: size
    height: size
    highlighted: pressed

    Image {
        id: defaultImage

        width: size
        height: size
        sourceSize.width: size
        sourceSize.height: size

        source: drawIcon ? "image://theme/icon-launcher-dummy" : ""
        visible: false
    }

    Component {
        id: sourceImageComponent

        Image {
            width: size
            height: size
            sourceSize.width: size
            sourceSize.height: size
            visible: false

            source: {
                if (_icon.indexOf(':/') !== -1 || _icon.indexOf("data:image/png;base64") === 0) {
                    return _icon
                } else if (_icon.indexOf('/') === 0) {
                    return 'file://' + _icon
                } else if (_icon.length) {
                    return 'image://theme/' + _icon
                } else {
                    return ""
                }
            }
        }
    }

    // Used to support placeholder drawing only,
    // for example: placeholder in LauncherGridFolder
    Loader {
        id: sourceImageLoader

        active: true
        sourceComponent: drawIcon ? sourceImageComponent : null
    }

    ShaderEffect {
        id: shader

        readonly property Image source: item.image

        // Pixel alignment for source image
        readonly property real fixedOutlineWidth: 2 * Math.round(outlineWidth * 0.5)
        readonly property real imageScale: width / (item.size * textureScale)
        readonly property real outlineScale: width / (item.size + outlineWidth * 2)
        readonly property real textureScale: drawIcon && scaleSource && item.image == sourceImageLoader.item ? 0.8 : 1.0

        readonly property real highlighted: item.highlighted ? 1 : 0 // no bools in shader
        readonly property real hardness: strengthFactor * 0.5
        readonly property real strengthFactor: item.size / strength
        readonly property real shadowRadius: dropShadow ? Theme.dp(10) : 0
        readonly property real shadowFactor: -(strengthFactor / shadowRadius * 2)
        readonly property real strength: 6
        readonly property size vstrength: Qt.size(strength, strength)

        readonly property string outlinedImageFragment: "
            float outline = clamp((1.0 - dot(pow(abs(outlineCoord), vstrength), vec2(1.0))) * hardness, 0.0, 1.0) * (1.0 - fill);
            vec4 filteredImage = outlineColor * outline + (image + fillColor * (1.0 - image.a)) * fill;"

        readonly property string maskedImageFragment: "
            vec4 filteredImage = (image + fillColor * (1.0 - image.a)) * fill;"

        readonly property string emptyImageFragment: "
            vec4 filteredImage = fillColor * fill;"

        readonly property string scaledImageFragment: "
            vec2 topLeftClip = step(vec2(-textureScale), fillCoord);
            vec2 bottomRightClip = step(vec2(textureScale), fillCoord);
            float clip = max(0.0, topLeftClip.x * topLeftClip.y - bottomRightClip.x - bottomRightClip.y);
            vec4 image = texture2D(source, imageCoord) * clip;"

        readonly property string cropImageFragment: "
            vec4 image = texture2D(source, imageCoord);"

        readonly property string highlightImageFragment: "
            vec4 highlight = highlightColor * highlighted * fill;
            filteredImage = highlight + filteredImage * (1.0 - highlight.a);"

        readonly property string shadowImageFragment: "
            vec4 shadow = shadowColor * clamp(1.0 / (1.0 + pow(3.0, sourceFill * shadowFactor)), 0.0, 1.0);
            filteredImage += shadow * (1.0 - filteredImage.a);"

        property bool dropShadow
        property real outlineWidth
        property color fillColor: "white"
        property color highlightColor: Theme.rgba(Theme.highlightBackgroundColor, 0.4)
        property color outlineColor: Theme.colorScheme === Theme.LightOnDark
                                        ? Theme.rgba("black", 0.2)
                                        : Theme.rgba("white", 0.4)

        property color shadowColor: Theme.rgba("black", 0.1)

        anchors {
            fill: parent
            margins: -(fixedOutlineWidth + shadowRadius)
        }

        vertexShader: "
            precision lowp float;

            uniform highp mat4 qt_Matrix;
            attribute highp vec4 qt_Vertex;
            attribute highp vec2 qt_MultiTexCoord0;
            varying vec2 imageCoord;
            varying vec2 fillCoord;
            varying vec2 outlineCoord;
            uniform float imageScale;
            uniform float outlineScale;
            uniform float outlineWidth;
            uniform float width;

            uniform vec4 qt_SubRect_source;

            vec2 scaledCoord(vec2 coord, float value) {
                return (coord - vec2(0.5)) * value + vec2(0.5);
            }

            void main() {
                outlineCoord = (2.0 * scaledCoord(qt_MultiTexCoord0, outlineScale) - 1.0);
                fillCoord = outlineCoord / (1.0 - 2.0 * outlineWidth / width);
                imageCoord = qt_SubRect_source.xy + qt_SubRect_source.zw * scaledCoord(qt_MultiTexCoord0, imageScale);
                gl_Position = qt_Matrix * qt_Vertex;
            }
        "

        fragmentShader: "
            precision lowp float;

            varying vec2 imageCoord;
            varying vec2 fillCoord;
            varying vec2 outlineCoord;

            uniform float qt_Opacity;
            uniform sampler2D source;

            uniform vec4 fillColor;
            uniform vec4 outlineColor;
            uniform vec4 highlightColor;
            uniform vec4 shadowColor;
            uniform vec2 vstrength;
            uniform float hardness;
            uniform float highlighted;
            uniform float textureScale;
            uniform float shadowFactor;

            void main() {
                float sourceFill = (1.0 - dot(pow(abs(fillCoord), vstrength), vec2(1.0)));
                float fill = clamp(sourceFill * hardness, 0.0, 1.0);

                %1 // image, missing when drawIcon==false
                %2 // filteredImage
                %3 // filteredImage + highlight
                %4 // filteredImage + shadow

                gl_FragColor = filteredImage * qt_Opacity;
            }
        ".arg(drawIcon ? (scaleSource ? scaledImageFragment : cropImageFragment) : "")
         .arg(drawIcon ? (outlineWidth > 0 ? outlinedImageFragment : maskedImageFragment) : emptyImageFragment)
         .arg(highlighted ? highlightImageFragment : "")
         .arg(dropShadow ? shadowImageFragment : "")
    }
}
