/****************************************************************************
**
** Copyright (c) 2022 Open Mobile Platform LLC.
**
** License: Proprietary
**
****************************************************************************/

import QtQuick 2.0
import Sailfish.Silica 1.0

Item {
    id: indicator

    property alias icon: icon
    property int edge: Qt.TopEdge
    property bool dragEnabled
    property bool pinned
    property bool touchActive
    property bool layerActive
    property bool layerEnabled

    property real progress
    property real progressThreshold: 1.0

    property real indicatorWidth: Theme.iconSizeMedium * 2
    property real indicatorHeight: Theme.paddingSmall
    property Item mainIndicator
    property Item edgeIndicator

    property real _bottomPadding: edge == Qt.BottomEdge
                                  ? Theme.paddingSmall * (dragEnabled ? 2 : 0.5)
                                  : 0
    property real _offset

    y: (edge == Qt.BottomEdge ? parent.height - height : 0)
       + _offset * (edge == Qt.BottomEdge ? -1 : 1)
    width: parent.width
    height: column.height + icon.height + _bottomPadding

    opacity: layerActive || !layerEnabled ? 0 : 1

    Behavior on opacity { FadeAnimation {} }

    Icon {
        id: icon

        anchors {
            top: edge == Qt.TopEdge ? column.bottom : undefined
            bottom: edge == Qt.BottomEdge ? column.top : undefined
            horizontalCenter: column.horizontalCenter
        }
    }

    Column {
        id: column

        y: edge == Qt.BottomEdge
           ? parent.height - height - _bottomPadding
           : 0
        width: parent.width

        spacing: Theme.paddingSmall / 2

        Repeater {
            model: !dragEnabled && edge == Qt.BottomEdge ? 2 : 1

            Rectangle {
                id: rect

                property bool widthAnimationsEnabled
                property bool xAnimationsEnabled

                x: (parent.width - width) / 2
                width: indicatorWidth
                height: indicatorHeight

                radius: height / 2
                color: pinned ? Theme.highlightColor : Theme.primaryColor

                Behavior on x {
                    enabled: rect.xAnimationsEnabled
                    NumberAnimation { easing.type: Easing.OutQuad; duration: 80 }
                }

                Behavior on width {
                    enabled: rect.widthAnimationsEnabled
                    NumberAnimation { easing.type: Easing.OutQuad; duration: 80 }
                }

                Component.onCompleted: {
                    if (!mainIndicator) {
                        mainIndicator = rect
                    } else {
                        edgeIndicator = rect
                    }
                }
            }
        }
    }

    states: [
        State {
            name: "active"
            when: indicator.touchActive
            PropertyChanges {
                target: indicator
                _offset: Math.min(1.0, indicator.progress / indicator.progressThreshold)
                         * Theme.paddingMedium
            }
        },
        State {
            name: "inactive"
            when: !indicator.touchActive
            PropertyChanges {
                target: indicator
                _offset: 0
            }
        }
    ]

    transitions: [
        Transition {
            from: "*"
            to: "inactive"
            NumberAnimation {
                property: "_offset"
                duration: 500
                easing.type: Easing.OutQuad
            }
        }
    ]
}

