/*
 * SPDX-FileCopyrightText: 2024-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.5
import Sailfish.Silica 1.0
import Sailfish.Silica.private 1.0
import Sailfish.Gallery 1.0
import Sailfish.Gallery.private 1.0

ZoomableFlickable {
    id: flickable

    property alias source: photo.source

    property string mimeType
    readonly property bool isSvg: mimeType === "image/svg+xml"

    property bool active: true
    readonly property bool _active: active || viewMoving
    readonly property bool error: photo.status == Image.Error
    readonly property alias imageMetaData: metadata

    property alias photo: photo
    property alias largePhoto: largePhoto
    readonly property real zoomStep: 1.2

    signal clicked
    signal doubleClicked

    contentRotation: -metadata.orientation
    scrollDecoratorColor: Theme.lightPrimaryColor

    zoomEnabled: photo.status == Image.Ready
    maximumZoom: Math.max(Screen.width, Screen.height) / 200
                 * Math.max(1.0, photo.implicitWidth > 0 ? largePhoto.implicitHeight / photo.implicitHeight
                                                         : 1.0)
    on_ActiveChanged: if (!_active) resetZoom()

    implicitContentWidth: photo.implicitWidth
    implicitContentHeight: photo.implicitHeight

    AnimatedImage {
        id: photo
        property var errorLabel
        objectName: "zoomableImage"

        anchors.fill: parent
        smooth: !(movingVertically || movingHorizontally)
        fillMode: Image.PreserveAspectFit
        asynchronous: true
        cache: false
        visible: !isSvg

        onStatusChanged: {
            if (status == Image.Error) {
                errorLabel = errorLabelComponent.createObject(flickable)
            }
        }

        onSourceChanged: {
            if (errorLabel) {
                errorLabel.destroy()
                errorLabel = null
            }

            resetZoom()
        }

        opacity: status == Image.Ready ? 1 : 0
        Behavior on opacity { FadeAnimation{} }
    }

    Image {
        id: largePhoto

        anchors.fill: parent
        source: isSvg ? photo.source : ""
        visible: isSvg
        asynchronous: true
        cache: false

        //! TODO: These are magic numbers to me. See src/private/ZoomableImage.qml for possible explanation
        sourceSize {
            width: 3264
            height: 3264
        }
    }

    Item {
        width: flickable.transpose ? parent.height : parent.width
        height: flickable.transpose ? parent.width : parent.height

        anchors.centerIn: parent
        rotation: -flickable.contentRotation

        MouseArea {
            x: width > parent.width
               ? (parent.width - width) / 2
               : flickable.contentX + Theme.paddingLarge
            y: height > parent.height
               ? (parent.height - height) / 2
               : flickable.contentY + Theme.paddingLarge
            width: flickable.width - (2 * Theme.paddingLarge)
            height: flickable.height - (2 * Theme.paddingLarge)

            onClicked: flickable.clicked()
            onDoubleClicked: flickable.doubleClicked()
            onWheel: {
                if (wheel.modifiers && Qt.ControlModifier) {
                    const zoomFactor = wheel.angleDelta.y > 0 ? zoomStep : 1.0 / zoomStep;
                    const localPos = mapToItem(flickable.contentItem, wheel.x, wheel.y)

                    flickable.zoomIn(
                        zoomFactor,
                        localPos,
                        localPos
                    )
                }
            }
        }
    }

    ImageMetadata {
        id: metadata

        source: photo.source
        autoUpdate: false
    }

    BusyIndicator {
        running: photo.status === Image.Loading && !delayBusyIndicator.running
        size: BusyIndicatorSize.Large
        anchors.centerIn: parent
        parent: flickable

        Timer {
            id: delayBusyIndicator
            running: photo.status === Image.Loading
            interval: 1000
        }
    }

    Component {
        id: errorLabelComponent

        ErrorItem { }
    }

    Component.onCompleted: {
        if (!mimeType) {
            console.warn("The mime type for ImageViewer is not set! Display issues may arise.")
        }
    }
}
