/*
 * SPDX-FileCopyrightText: 2012-2019 Jolla Ltd
 * SPDX-FileCopyrightText: 2019-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.6
import Sailfish.Silica 1.0
import Sailfish.Silica.private 1.0 as Private
import Sailfish.Contacts 1.0
import org.nemomobile.contacts 1.0

FocusScope {
    id: root

    property alias icon: fieldIcon
    property alias value: inputField.text
    property int detailType: -1
    property alias placeholderText: inputField.placeholderText
    property alias inputMethodHints: inputField.inputMethodHints
    property bool canRemove
    property bool canEdit: true

    property alias buttonModeText: addFieldButton.text
    property bool buttonMode
    property alias showIconWhenEditing: addFieldButton.showIconWhenOffscreen

    property bool animate
    property alias animationDuration: addFieldButton.animationDuration
    property alias leftMargin: inputField.textLeftMargin
    property alias rightMargin: inputField.textRightMargin
    readonly property alias hasFocus: inputField.activeFocus
    property string keyboardEnterIcon: "image://theme/icon-m-enter-next"

    property int detailSubType
    property int detailLabel
    property var detailSubTypeModel
    property bool showDetailLabelCombo

    property bool _addButtonClicked
    property bool _textWasEntered

    property ContactDetailSuggestions suggestions
    property int suggestionField: ContactDetailSuggestions.None

    signal clickedInButtonMode()
    signal enteredButtonMode()
    signal modified()
    signal removeClicked()
    signal accepted()
    signal editingFinished()

    signal detailSubTypeModified(int type, int subType)
    signal detailLabelModified(int label)

    function forceActiveFocus() {
        inputField.forceActiveFocus()
    }

    function clearFocus() {
        inputField.focus = false
    }

    width: parent.width
    height: root.buttonMode ? addFieldButton.height + addFieldButton.anchors.topMargin
                            : inputField.height + inputField.anchors.topMargin + ContactsUtil.__editorVerticalPadding
    Item {
        id: fieldIconItem

        anchors {
            top: parent.top
            topMargin: ContactsUtil.__editorIconsVerticalPadding
            left: parent.left
            leftMargin: Theme.paddingLarge
        }

        width: Theme.dp(48)
        height: width
        opacity: !root.buttonMode ? 1.0 : 0.0

        Icon {
            id: fieldIcon

            anchors.left: parent.left
            highlighted: root.highlighted
        }
    }

    TextField {
        id: inputField

        anchors {
            top: parent.top
            topMargin: Theme.paddingMedium
            left: fieldIconItem.right
            leftMargin: Theme.paddingLarge
            right: parent.right
        }

        readOnly: !root.enabled || !root.canEdit
        highlighted: activeFocus
        opacity: !root.buttonMode ? 1.0 : 0.0

        EnterKey.iconSource: root.keyboardEnterIcon
        EnterKey.onClicked: root.accepted()

        focus: true

        onTextChanged: {
            if (activeFocus) {
                // Once text changes, avoid returning to button mode.
                root._textWasEntered = true
                root.modified()
            }
        }

        onActiveFocusChanged: {
            if (activeFocus && root.suggestions) {
                root.suggestions.field = root.suggestionField
                if (root.suggestionField !== ContactDetailSuggestions.None) {
                    root.suggestions.partialText = Qt.binding(function() {
                        return inputField.text
                    })
                    root.suggestions.inputItem = inputField._editor
                }
            }

            if (!activeFocus) {
                root.editingFinished()
            }
        }

        label: !!root.detailSubTypeModel ? "" : placeholderText
        labelComponent: (!!root.detailSubTypeModel || root.showDetailLabelCombo)
                        ? miniCombosComponent
                        : inputField.defaultLabelComponent

        Component {
            id: miniCombosComponent

            Row {
                id: miniCombos

                readonly property real initialHeight: detailSubTypeCombo.contentHeight
                readonly property int maximumWidth: inputField.width - Theme.paddingSmall

                MiniComboBox {
                    id: detailSubTypeCombo

                    width: ContactsUtil.calcMiniComboBoxWidth(implicitWidth,
                                                              detailLabelCombo.implicitWidth,
                                                              maximumWidth)
                    label: root.placeholderText
                    visible: !!root.detailSubTypeModel
                    menu: DetailSubTypeMenu {
                        model: root.detailSubTypeModel
                        currentSubType: root.detailSubType

                        onCurrentIndexChanged: detailSubTypeCombo.currentIndex = currentIndex
                        onSubTypeClicked: root.detailSubTypeModified(type, subType)
                    }
                }

                MiniComboBox {
                    id: detailLabelCombo

                    width: ContactsUtil.calcMiniComboBoxWidth(implicitWidth,
                                                              detailSubTypeCombo.implicitWidth,
                                                              maximumWidth)

                    // When 'None' is selected, show 'Select label' instead.
                    value: (!!currentItem && currentItem.text == ContactsUtil.getNoLabelText())
                           ? ContactsUtil.getSelectLabelText()
                           : currentItem.text

                    menu: DetailLabelMenu {
                        currentLabel: root.detailLabel
                        detailType: root.detailType

                        onCurrentIndexChanged: detailLabelCombo.currentIndex = currentIndex
                        onLabelClicked: root.detailLabelModified(label)
                    }
                }
            }
        }

        Behavior on opacity { FadeAnimation {} }
    }

    AddFieldButton {
        id: addFieldButton

        anchors {
            top: parent.top
            topMargin: Theme.paddingMedium
            left: fieldIconItem.right
            leftMargin: Theme.paddingLarge
            right: parent.right
        }

        x: parent.width + width + Theme.paddingMedium
        offscreen: !root.buttonMode
        opacity: visible ? 1.0 : 0.0
        visible: root.buttonMode
        highlighted: inputField.activeFocus || down

        onClicked: {
            animate = true
            root._addButtonClicked = true

            if (root.buttonMode) {
                root.clickedInButtonMode()
            } else {
                inputField.forceActiveFocus()
            }
        }

        onEnteredButtonMode: root.enteredButtonMode()
        onExitedButtonMode: inputField.forceActiveFocus()

        Behavior on opacity { FadeAnimation {} }
    }
}
