/*
 * SPDX-FileCopyrightText: 2023-2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.6
import Sailfish.Silica 1.0
import Sailfish.Contacts 1.0
import org.nemomobile.contacts 1.0

MultiTypeFieldEditor {
    id: root

    function populateFieldEditor() {
        detailModel.reload(contact[propertyAccessor])
        detailSubTypeModel.reload(allowedTypes, allowedSubTypes)

        // User can only add notes if there are zero notes for this contact. This follows the
        // style of typical contact services that only allow one note per contact. However, if
        // a contact has multiple notes (e.g. synced from some service that does allow multiple
        // notes to be created) then these will be shown.
        if (detailModel.count === 0) {
            addEmptyField()
        }
    }

    //: Add a note for this contact
    //% "Add note"
    fieldAdditionText: qsTrId("contacts-bt-contact_add_note")
    titleIconSource: "image://theme/icon-splus-note"

    propertyAccessor: 'noteDetails'
    valueField: 'note'
    allowedTypes: [ Person.NoteType ]
    canChangeLabelType: false
    showTitleLabel: true
    titleLabelText: ContactsUtil.getNameForDetailType(Person.NoteType)

    fieldDelegate: Item {
        id: noteDelegate

        width: parent.width
        height: inputField.height

        TextArea {
            id: inputField

            anchors {
                top: parent.top
                topMargin: Theme.dp(16)
                left: parent.left
                leftMargin: Theme.iconSizeSmallPlus + Theme.paddingLarge
                right: parent.right
            }
            height: implicitHeight  // explicit binding needed to trigger height Behavior
            textLeftMargin: Theme.paddingLarge

            activeFocusOnTab: root.expanded
            highlighted: activeFocus
            opacity: root.expanded ? 1.0 : 0.0
            enabled: root.expanded
            focus: root.initialFocusIndex === model.index

            text: model.value
            label: model.name
            placeholderText: model.name

            onTextChanged: {
                if (activeFocus) {
                    root.detailModel.userModified = true
                    root.detailModel.setProperty(model.index, "value", text)
                }
            }

            // Don't autoscroll to beneath the DialogHeader
            VerticalAutoScroll.topMargin: Theme.itemSizeLarge

            Behavior on height {
                id: noteHeightAnimation

                enabled: false

                NumberAnimation {
                    duration: root.animationDuration
                    easing.type: Easing.InOutQuad
                    onRunningChanged: noteHeightAnimation.enabled = false
                }
            }

            Behavior on opacity { FadeAnimation {} }
        }
    }
}

