/*
 * SPDX-FileCopyrightText: Copyright 2022-2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 *
 * Copyright (c) 2019 - 2020 Jolla Ltd.
 *
 * License: Proprietary
*/

import QtQuick 2.0
import Sailfish.Silica 1.0
import Sailfish.Contacts 1.0
import org.nemomobile.contacts 1.0

BaseEditor {
    id: root

    property NicknameDetailModel nicknameModel
    property var flickable

    // This editor shows name fields in this order: first name, nickname, middle name, last name.
    // Nicknames cannot be added from this editor, but are shown if they exist when the UI is
    // first shown. User can add nicknames from the 'Add details' InfoEditor.
    readonly property var _fields: [
        { "type": Person.FirstNameType, "propertyName": "firstName", expandedProperty: false },
        { "type": Person.MiddleNameType, "propertyName": "middleName", expandedProperty: true },
        { "type": Person.LastNameType, "propertyName": "lastName", expandedProperty: false },
        { "type": Person.NicknameType, "propertyName": "nicknameDetails", expandedProperty: true }
    ]

    function populateFieldEditor() {
        detailModel.clear()

        for (var i = 0; i < root._fields.length; ++i) {
            var detail = root._fields[i]
            var value = ""

            if (detail.type === Person.NicknameType) {
                if (nicknameModel.count > 0) {
                    value = nicknameModel.get(0).value
                }
            } else {
                value = contact[detail.propertyName]
            }
            var expand = value.length > 0 ? false : detail.expandedProperty
            detailModel.append(_nameProperties(detail.type, detail.propertyName, value, -1, expand))
        }
    }

    function aboutToSave() {
        // Copy nickname additions to nicknameModel
        for (var i = 0; i < detailModel.count; ++i) {
            var detail = detailModel.get(i)
            if (detail.type === Person.NicknameType) {
                nicknameModel.set(0, detail)
                nicknameModel.userModified = true
            }
        }

        // Ignore nickname changes, as those are handled in ContactDetailDialog.
        detailModel.copySingleTypeDetailChanges(contact, Person.NicknameType)
    }

    function _nameProperties(type, propertyName, value, sourceIndex, expandedProperty) {
        var props = {
            "type": type,
            "subType": Person.NoSubType,
            "label": Person.NoLabel,
            "name": ContactsUtil.getNameForDetailType(type),
            "propertyName": propertyName,
            "value": value,
            "sourceIndex": sourceIndex,
            "expandedProperty": expandedProperty
        }
        return props
    }

    fieldDelegate: EditorFieldDelegate {
        id: nameDelegate

        editor: root
        leftMargin: 0
        rightMargin: Theme.paddingLarge
        focus: root.initialFocusIndex === model.index
        canEdit: true
        icon.source: index === 0 ? "image://theme/icon-splus-contact" : ""
        expandedProperty: model.expandedProperty
        height: !visible ? 0 : Theme.itemSizeMedium + ContactsUtil.__editorVerticalPadding
        opacity: visible ? 1.0 : 0.0

        onModified: {
            var trimmedValue = value.trim()
            root.detailModel.setProperty(model.index, "value", trimmedValue)
            if (model.type === Person.NicknameType) {
                var nicknameIndex = nicknameModel.findNicknameWithSourceIndex(model.sourceIndex)
                if (nicknameIndex >= 0) {
                    nicknameModel.userModified = true
                    nicknameModel.setProperty(0, "value", trimmedValue)
                }
            }

            root.hasContent = trimmedValue.length > 0 || root.testHasContent()
        }

        onRemoveClicked: {
            if (model.type === Person.NicknameType && !canEdit) {
                root.animateAndRemove(model.index, nameDelegate)

                var nicknameIndex = nicknameModel.findNicknameWithSourceIndex(model.sourceIndex)
                if (nicknameIndex >= 0) {
                    nicknameModel.userModified = true
                    nicknameModel.setProperty(0, "value", "")
                }
            }
        }
    }
}
