/*
 * SPDX-FileCopyrightText: 2023-2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.6
import Sailfish.Contacts 1.0 as SailfishContacts
import Sailfish.Silica 1.0

Item {
    id: root

    readonly property int animationDuration: 600
    property bool expanded: false
    property bool showExpandButton: true
    property bool showBackground: true

    default property alias children: contentContainer.data
    property alias _contentContainer: contentContainer

    width: parent.width
    height: contentContainer.height + Theme.paddingLarge * 2

    Component.onDestruction: {
        expanded = false
    }

    Component.onCompleted: {
        root.state = "shrinked"
    }

    onExpandedChanged: {
        if (expanded) {
            root.state = "expanded"
        } else {
            root.state = "shrinked"
        }
    }

    Behavior on expanded { FadeAnimation { duration: root.animationDuration } }
    Behavior on height { FadeAnimation { duration: root.animationDuration } }

    Rectangle {
        anchors.fill: parent
        visible: root.showBackground
        color: Theme.rgba(Theme.highlightColor, 0.1)
        radius: Theme.dp(12)
    }

    BackgroundItem {
        id: iconItem

        anchors {
            top: parent.top
            topMargin: Theme.paddingLarge + SailfishContacts.ContactsUtil.__editorIconsVerticalPadding
            right: parent.right
            rightMargin: Theme.paddingLarge
            leftMargin: Theme.paddingLarge
            bottomMargin: Theme.paddingLarge
        }

        width: Theme.iconSizeSmallPlus
        height: Theme.iconSizeSmallPlus
        visible: root.showExpandButton

        Icon {
            id: icon

            anchors.centerIn: parent
            width: Theme.iconSizeSmallPlus
            height: Theme.iconSizeSmallPlus

            source: root.expanded ? "image://theme/icon-splus-up" : "image://theme/icon-splus-down"
            highlighted: iconItem.down
            Behavior on source { FadeAnimation { duration: root.animationDuration } }
        }

        onClicked: {
            root.expanded = !root.expanded

            for (var key in root.children) {
                var item = root.children[key]

                if (!!item.expandedProperty && item.expandedProperty) {
                    item.visible = root.expanded
                }
            }
        }
    }

    Column {
        id: contentContainer

        anchors {
            top: parent.top
            topMargin: Theme.paddingLarge
            left: parent.left
            right: iconItem.left
            bottomMargin: Theme.paddingLarge
        }

        height: contentContainer.contentHeight

        spacing: Theme.paddingMedium

        Behavior on height { PropertyAnimation { duration: root.animationDuration } }
        Behavior on opacity { FadeAnimation { duration: root.animationDuration } }
    }

}
