/*
 * SPDX-FileCopyrightText: 2023-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.6
import Sailfish.Silica 1.0
import Sailfish.Contacts 1.0
import org.nemomobile.contacts 1.0

MultiTypeFieldEditor {
    id: root

    property int _dummyBirthdaySubType: -9999
    property bool hasBirthdayField: _birthdayFieldIndex() >= 0
    property date invalidDate: new Date(NaN)
    property bool isValideDate: true
    property bool hasValidDate: testDate()

    // override BaseEditor testHasContent()
    function testHasContent(listModel) {
        listModel = listModel || detailModel
        return listModel.count > 1
    }

    function testDate() {
        for (var i = 0; i < detailModel.count; ++i) {
            if (!isValideDate && (i != detailModel.count - 1) && isNaN(detailModel.get(i).value)) {
                return false
            }
        }

        return true
    }

    function populateFieldEditor() {
        detailModel.emptyValue = new Date(Number.NaN)

        // Add dates from Person::anniversaryDetails.
        detailModel.reload(contact[propertyAccessor])

        // Add birthday from Person::birthday.
        if (!isNaN(contact.birthday)) {
            var birthdayDetails = contact['birthdayDetail']
            var properties = {
                "type": Person.BirthdayType,
                "subType": _dummyBirthdaySubType,
                "name": ContactsUtil.getNameForDetailType(Person.BirthdayType),
                "value": contact.birthday,
                "sourceIndex": -1,
                "showYear": birthdayDetails.showYear
            }
            // Show birthday before other date values.
            detailModel.insert(0, properties)
        }

        // Add sub-type options for dates by combining anniversary sub-types with an extra
        // 'birthday' type. To prevent the user from setting more than one data as a birthday,
        // use two models for the date type combo menus: one with 'Birthday', and one without,
        // and show the correct one  depending on whether a birthday has already been selected.
        detailSubTypeModel.reload(allowedTypes, allowedSubTypes)
        detailSubTypeModelWithBirthday.reload(allowedTypes, allowedSubTypes)
        detailSubTypeModelWithBirthday.insert(1, {
                                                  "type": Person.BirthdayType,
                                                  "subType": _dummyBirthdaySubType,
                                                  "name": ContactsUtil.getNameForDetailType(Person.BirthdayType)
                                              })

        // Add an empty field to act as the 'Add date' button.
        if (detailModel.count === 0) {
            var properties = {
                "type": Person.BirthdayType,
                "subType": _dummyBirthdaySubType,
                "name": ContactsUtil.getNameForDetailType(Person.BirthdayType),
                "value": root.invalidDate,
                "sourceIndex": -1,
                "showYear": true
            }
            detailModel.append(properties)
        } else {
            root.addEmptyField()
        }
    }

    function aboutToSave() {
        if (detailModel.userModified) {
            // Copy birthday value to Person::birthday.
            var birthdayIndex = _birthdayFieldIndex()
            contact.birthday = birthdayIndex >= 0 ? detailModel.get(birthdayIndex).value : detailModel.emptyValue

            if (!isNaN(contact.birthday)) {
                contact.setYearVisibilityForBirthday(detailModel.get(birthdayIndex).showYear)
            }

            // Copy non-birthday dates from detailModel to Person::anniversaryDetails.
            detailModel.copyMultiTypeDetailChanges(contact, propertyAccessor, Person.BirthdayType)
        }
    }

    function _birthdayFieldIndex() {
        for (var i = 0; i < detailEditors.count; ++i) {
            if (root.detailEditors.itemAt(i).isBirthday) {
                return i
            }
        }
        return -1
    }

    //: Add a date (e.g. a wedding anniversary date) for this contact
    //% "Add date"
    fieldAdditionText: qsTrId("contacts-bt-contact_add_date")
    titleIconSource: "image://theme/icon-splus-date"

    propertyAccessor: 'anniversaryDetails'
    valueField: 'originalDate'
    allowedTypes: [ Person.AnniversaryType ]
    subTypesExclusive: true
    allowedSubTypes: {
        var subTypes = {}
        subTypes[Person.AnniversaryType] = [
                    Person.AnniversarySubTypeWedding,
                    Person.AnniversarySubTypeEngagement,
                    Person.AnniversarySubTypeHouse,
                    Person.AnniversarySubTypeEmployment,
                    Person.AnniversarySubTypeMemorial
                ]
        return subTypes
    }

    canChangeLabelType: false
    showTitleLabel: true
    titleLabelText: ContactsUtil.getNameForDetailType(Person.AnniversaryType)

    fieldDelegate: Item {
        id: dateDelegate

        readonly property int dateIndex: model.index
        readonly property int isBirthday: model.type === Person.BirthdayType
        readonly property int dateSubType: model.subType

        readonly property string dateFormat: showYearSwitch.checked ? "dd.MM.yyyy" : "dd.MM"

        property date currDate: new Date(NaN)
        property bool fullDateIsSet

        function _changeDate(fieldIndex, currentValue) {
            Qt.inputMethod.hide()
            var obj = pageStack.animatorPush("Sailfish.Silica.DatePickerDialog", { date: currentValue, _showYearSelectionFirst: false })
            obj.pageCompleted.connect(function(dialog) {
                dialog.accepted.connect(function() {
                    var dText = ContactsUtil.getDateButtonText(dateDelegate.dateFormat, dialog.date)
                    dateLabel.text = dText
                    dateDelegate.currDate = dialog.date
                    root.detailModel.userModified = true
                    root.detailModel.setProperty(fieldIndex, "value", dateDelegate.currDate)
                    root.detailModel.setProperty(fieldIndex, "showYear", showYearSwitch.checked)

                    if (dateIndex === root.detailModel.count - 1) {
                        root.addEmptyField()
                    }
                })
            })
        }

        function exitButtonMode() {
            addDateButton.offscreen = true
        }

        function checkValid() {
            if (dateLabel.text.length === 0) {
                root.isValideDate = true
            } else {
                root.isValideDate = dateLabel.acceptableInput
            }
        }

        width: parent.width
        height: fieldIconItem.height +
                (addDateButton.offscreen
                 ? dateDisplay.height
                 : addDateButton.anchors.topMargin)

        visible: root.expanded

        Component.onCompleted: {
            if (!isNaN(model.value)) {
                exitButtonMode()
            }

            if (root.detailModel.count === 1) {
                addDateButton.offscreen = true
            }
        }

        Behavior on height {
            enabled: root.populated

            NumberAnimation {
                duration: root.animationDuration
                easing.type: Easing.InOutQuad
            }
        }

        //Header
        Item {
            id: fieldIconItem

            anchors {
                top: parent.top
                left: parent.left
                leftMargin: Theme.paddingLarge
            }
            width: Theme.iconSizeSmallPlus
            height: dateIndex !== 0 ? Theme.dp(64) : 0
            opacity: root.showTitleLabel ? 1.0 : 0.0
            visible: dateIndex !== 0 && addDateButton.offscreen

            Icon {
                id: fieldIcon

                anchors {
                    left: parent.left
                    verticalCenter: parent.verticalCenter
                }
                highlighted: root.expanded || root.highlighted
                source: root.titleIconSource
            }
        }

        Label {
            id: titleLabel

            anchors {
                verticalCenter: fieldIconItem.verticalCenter
                left: fieldIconItem.right
                leftMargin: Theme.paddingLarge
                right: clearButton.left
                rightMargin: Theme.paddingSmall
            }

            visible: root.showTitleLabel && dateIndex !== 0 && addDateButton.offscreen
            highlighted: root.expanded || root.highlighted

            text: root.titleLabelText
        }

        IconButton {
            id: clearButton

            anchors {
                left: parent.right
                verticalCenter: titleLabel.verticalCenter
            }

            visible: dateIndex !== 0 && addDateButton.offscreen
            width: Theme.iconSizeSmallPlus
            height: Theme.iconSizeSmallPlus
            icon.source: "image://theme/icon-splus-remove"
            highlighted: true

            onClicked: {
                root.detailModel.userModified = true
                if (!root.animateAndRemove(model.index, dateDelegate)) {
                    addDateButton.offscreen = false
                    root.detailModel.setProperty(model.index, "value", root.invalidDate)
                }

                if (model.index === root.detailModel.count - 1) {
                    root.addEmptyField()
                }
            }
        }


        Item {
            id: dateDisplay

            anchors {
                top: dateIndex === 0 ? parent.top : fieldIconItem.bottom
                left: parent.left
                leftMargin: Theme.iconSizeSmallPlus + Theme.paddingLarge
                right: parent.right
            }

            height: dateTypeCombo.height + dateLabel.height + showYearSwitch.height + Theme.paddingSmall
            enabled: addDateButton.offscreen
            opacity: addDateButton.revealedContentOpacity

            MiniComboBox {
                id: dateTypeCombo

                anchors {
                    top: parent.top
                    topMargin: Theme.paddingSmall
                    left: parent.left
                    leftMargin: Theme.paddingLarge
                }
                menu: DetailSubTypeMenu {
                    model: dateDelegate.isBirthday || !root.hasBirthdayField
                           ? detailSubTypeModelWithBirthday
                           : root.detailSubTypeModel
                    currentSubType: dateDelegate.isBirthday
                                    ? _dummyBirthdaySubType
                                    : dateDelegate.dateSubType

                    onCurrentIndexChanged: {
                        dateTypeCombo.currentIndex = currentIndex
                    }

                    onSubTypeClicked: {
                        var birthdaySelected = (subType === _dummyBirthdaySubType)
                        root.setDetailType(dateDelegate.dateIndex, type, birthdaySelected ? Person.NoSubType : subType)
                    }
                }
            }

            TextField {
                id: dateLabel

                property int symbolsCount: 0
                property bool deleteOperation

                anchors {
                    top: dateTypeCombo.bottom
                    topMargin: Theme.paddingMedium
                    left: dateTypeCombo.left
                    right: parent.right
                    rightMargin: Theme.paddingLarge
                }

                text: {
                    dateDelegate.currDate = new Date(model.value)
                    if (!isNaN(currDate)) {
                        fullDateIsSet = true
                    }

                    return ContactsUtil.getDateText(model.value, dateDelegate.dateFormat)
                }

                labelVisible: !dateLabel.acceptableInput && text.length > 0

                //% "Incorrect date format"
                label: qsTrId("contacts-lt-incorrect_date_format")
                textLeftMargin: 0
                placeholderText: showYearSwitch.checked
                                   //% "dd.mm.yyyy"
                                 ? qsTrId("contacts-la-contact_date_full")
                                   //% "dd.mm"
                                 : qsTrId("contacts-la-contact_date_short")
                inputMethodHints: Qt.ImhDigitsOnly | Qt.ImhNoPredictiveText
                color: addDateButton.highlighted ? Theme.highlightColor : Theme.primaryColor
                validator: dateValidator

                rightItem: IconButton {
                    icon.source: "image://theme/icon-splus-date"
                    onClicked: {
                        _changeDate(model.index, model.value)
                    }
                }

                onActiveFocusChanged: {
                    if (activeFocus) {
                        cursorPosition = text.length
                    }
                }

                onTextChanged: {
                    if (symbolsCount != text.length) {
                        deleteOperation = symbolsCount > text.length
                        symbolsCount = text.length
                    }

                    const regex = /\d/g
                    text = text.replace(/,/g,".")
                    const position = text.length - 1

                    if (text.length > 0 && position < text.length && text[position].match(regex)) {
                        if (position === 1 && !deleteOperation) {
                            text = text.slice(0, 2) + "." + text.slice(2)
                        } else if (position === 4 && showYearSwitch.checked && !deleteOperation) {
                            text = text.slice(0, 5) + "." + text.slice(5)
                        }
                    }

                    checkValid()
                }

                onAcceptableInputChanged: {
                    if (dateLabel.acceptableInput && dateLabel.text.length > 0) {
                        var textDate = dateLabel.text

                        if (!showYearSwitch.checked) {
                            if (!fullDateIsSet) {
                                dateLabel.text = dateLabel.text.slice(0, 5)
                            }

                            if (isNaN(currDate)) {
                                var currYear = new Date()
                                textDate = textDate + "." +  currYear.getFullYear()
                            } else {
                                var year = currDate.getFullYear()
                                textDate = textDate + "." +  year
                            }
                        } else {
                            fullDateIsSet = true
                        }

                        currDate = ContactsUtil.getDate(showYearSwitch.checked, textDate, dateLabel.acceptableInput)
                        root.detailModel.userModified = true
                        root.detailModel.setProperty(dateDelegate.dateIndex, "value", dateDelegate.currDate)
                        root.detailModel.setProperty(dateDelegate.dateIndex, "showYear", showYearSwitch.checked)

                        if (dateIndex === root.detailModel.count - 1) {
                            root.addEmptyField()
                        }
                    } else if (dateLabel.text.length > 0) {
                        if (showYearSwitch.checked) {
                            fullDateIsSet = false
                        }

                        currDate = root.invalidDate
                        root.detailModel.userModified = true
                        root.detailModel.setProperty(dateDelegate.dateIndex, "value", dateDelegate.currDate)
                    }

                    checkValid()
                }

                TextMetrics {
                    id: metrics

                    text: dateLabel.text
                    font: dateLabel.font
                }

                Label {
                    x: metrics.advanceWidth
                    height: dateLabel.height
                    text: dateLabel.placeholderText.substring(dateLabel.length)
                    horizontalAlignment: dateLabel.horizontalAlignment
                    font: dateLabel.font
                    leftPadding: dateLabel.textLeftPadding
                    visible: dateLabel.length > 0
                    opacity: dateLabel.focus ? Theme.opacityLow : 1.0
                }

            }

            TextSwitch {
                id: showYearSwitch

                anchors {
                    top: dateLabel.bottom
                    topMargin: -Theme.paddingSmall
                    left: dateTypeCombo.left
                    //Move the switch icon slightly to the left to align it with the text field
                    leftMargin: Theme.dp(-20)
                }

                //% "Show year"
                text: qsTrId("contacts-bt-contact_show_year")
                checked: model.showYear
                _label.font.pixelSize: Theme.fontSizeSmall
            }

            DateValidator {
                id: dateValidator

                pattern: dateDelegate.dateFormat

                onPatternChanged: {
                    if (!isNaN(dateDelegate.currDate) && fullDateIsSet) {
                        const formattedDate = ContactsUtil.getDateButtonText(pattern, dateDelegate.currDate)
                        dateLabel.text = formattedDate
                    } else if (!dateLabel.acceptableInput && dateLabel.length > 0) {
                        if (dateLabel.length === 6 && !showYearSwitch.checked) {
                            dateLabel.text = dateLabel.text.slice(0, -1)
                        } else if (dateLabel.length === 5 && !showYearSwitch.checked) {
                            dateLabel.text = dateLabel.text + " "
                            dateLabel.text = dateLabel.text.slice(0, -1)
                        }
                    } else if (dateLabel.acceptableInput && dateLabel.length === 5 && showYearSwitch.checked) {
                        dateLabel.text = dateLabel.text + "."
                    }

                    if (dateLabel.acceptableInput && !isNaN(dateDelegate.currDate)) {
                        root.detailModel.userModified = true
                        root.detailModel.setProperty(dateDelegate.dateIndex, "value", dateDelegate.currDate)
                        root.detailModel.setProperty(dateDelegate.dateIndex, "showYear", showYearSwitch.checked)
                    }

                    checkValid()
                }
            }
        }

        AddFieldButton {
            id: addDateButton

            anchors {
                top: parent.top
                topMargin: Theme.paddingMedium
                left: parent.left
                leftMargin: Theme.iconSizeSmallPlus + Theme.paddingLarge * 2
                right: parent.right
            }

            text: root.fieldAdditionText
            showIconWhenOffscreen: false
            animate: root.ready
            opacity: enabled ? 1 : 0
            highlighted: down

            onClicked: {
                offscreen = true
                // Assign 'birthday' type if no other date is currently set as birthday
                root.detailModel.setProperty(model.index,
                                             "type",
                                             _birthdayFieldIndex() < 0 ? Person.BirthdayType : Person.AnniversaryType)
                root.detailModel.setProperty(dateDelegate.dateIndex, "showYear", true)
            }

            onEnteredButtonMode: {
                root.resetField(model.index)
            }
        }
    }

    DetailSubTypeModel {
        id: detailSubTypeModelWithBirthday
    }
}
