/*
 * SPDX-FileCopyrightText: 2012-2019 Jolla Ltd.
 * SPDX-FileCopyrightText: 2019-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.6
import Sailfish.Silica 1.0
import Sailfish.Silica.private 1.0 as SilicaPrivate
import Sailfish.Contacts 1.0
import org.nemomobile.contacts 1.0

Item {
    id: root

    property alias _contentContainer: contentColumn
    property alias _titleIconContainer: fieldIconItem

    property string fieldAdditionText
    property url fieldAdditionIcon
    property int inputMethodHints: Qt.ImhNone
    property alias detailEditors: detailsRepeater
    property alias fieldDelegate: detailsRepeater.delegate
    property int initialFocusIndex: -1

    property ContactDetailSuggestions suggestions

    property Person contact
    property var peopleModel
    property ListModel detailModel: DetailModel {}
    property int animationDuration: 350 // Same as button animation in AddFieldButton
    property bool populated
    property bool hasContent: testHasContent()

    readonly property bool ready: populated && _page && _page.status === PageStatus.Active

    readonly property Page _page: {
        var parentItem = root.parent
        while (parentItem) {
            if (parentItem.hasOwnProperty('__silica_page')) {
                return parentItem
            }
            parentItem = parentItem.parent
        }
        return null
    }

    property string titleLabelText
    property bool showTitleLabel
    property url titleIconSource
    property bool expanded: false
    property bool showExpandButton: true
    property bool showBackground: true
    default property alias children: detailsRepeater.data

    function animateAndRemove(detailIndex, item, customAnimationDuration) {
        if (detailIndex < 0 || detailIndex >= detailModel.count || !item) {
            console.warn("invalid index or item:", detailIndex, item)
            return false
        }

        if (detailModel.count - 1 === 0 && !detailEditors.itemAt(detailIndex).buttonMode) {
            // This is the last field remaining, so don't remove it.
            focus = true    // ensure vkb is closed
            detailModel.setProperty(detailIndex, "value", detailModel.emptyValue)
            return false
        }

        if (!!removalHeightAnimation.target) {
            // Just remove immediately instead of clashing with a current animation.
            detailModel.remove(detailIndex)
        } else {
            if (customAnimationDuration !== undefined) {
                removalHeightAnimation.duration = customAnimationDuration
            }
            removalHeightAnimation.target = item
            delayedRemove.detailIndex = detailIndex
            removalAnimation.start()
        }

        return true
    }

    function testHasContent(listModel) {
        listModel = listModel || detailModel
        for (var i = 0; i < listModel.count; ++i) {
            if (listModel.get(i).value.trim().length > 0) {
                return true
            }
        }

        return false
    }

    function getHeight() {
        return (contentColumn.visible ? contentColumn.implicitHeight +
                                        contentColumn.anchors.topMargin +
                                        contentColumn.anchors.bottomMargin
                                      : 0) +
                (root.showTitleLabel ? Math.max(fieldIconItem.height, titleLabel.implicitHeight) +
                                       fieldIconItem.anchors.topMargin * 2
                                     : 0 )
    }

    height: getHeight()
    width: parent.width

    Behavior on height {
        NumberAnimation {
            duration: 20
            easing.type: Easing.InOutQuad
        }
    }

    onContactChanged: {
        if (contact) {
            populateFieldEditor()
            populated = true
        }
    }

    onExpandedChanged: {
        for (var key = 0; key < detailModel.count; key++) {
            var item = detailModel.get(key)

            if (!!item.expandedProperty && item.expandedProperty) {
                detailsRepeater.itemAt(key).visible = root.expanded
            }
        }
    }

    ParallelAnimation {
        id: removalAnimation

        NumberAnimation {
            id: removalHeightAnimation
            property: "height"
            duration: root.animationDuration
            to: 0
            easing.type: Easing.InOutQuad
        }

        FadeAnimation {
            target: removalHeightAnimation.target
            duration: removalHeightAnimation.duration/2
            to: 0
        }

        onStopped: {
            // Remove the detail in the next event loop.
            delayedRemove.start()
        }
    }

    Timer {
        id: delayedRemove

        property int detailIndex: -1

        interval: 0

        onTriggered: {
            var item = detailsRepeater.itemAt(detailIndex)

            if (!!item && item === removalHeightAnimation.target) {
                removalHeightAnimation.target = null
                root.detailModel.remove(detailIndex)
            }
        }
    }

    Rectangle {
        anchors.fill: parent
        visible: root.showBackground
        color: Theme.rgba(Theme.highlightColor, 0.1)
        radius: Theme.dp(12)
    }

    Item {
        id: fieldIconItem

        anchors {
            top: parent.top
            topMargin: root._contentContainer.anchors.topMargin
            left: parent.left
            leftMargin: Theme.paddingLarge
        }

        width: Theme.iconSizeSmallPlus
        height: Theme.dp(64)
        opacity: root.showTitleLabel ? 1.0 : 0.0

        Icon {
            id: fieldIcon

            anchors {
                left: parent.left
                verticalCenter: parent.verticalCenter
            }

            highlighted: root.expanded || root.highlighted
            source: root.titleIconSource
        }
    }

    Label {
        id: titleLabel

        anchors {
            verticalCenter: fieldIconItem.verticalCenter
            left: fieldIconItem.right
            leftMargin: Theme.paddingLarge
            right: iconItem.left
            rightMargin: Theme.paddingSmall
        }

        visible: root.showTitleLabel
        highlighted: root.expanded
        text: root.titleLabelText

        MouseArea {
            anchors.fill: parent
            hoverEnabled: true
            cursorShape: Qt.PointingHandCursor

            onClicked: {
                root.expanded = !root.expanded
            }
        }
    }

    Column {
        id: contentColumn

        anchors {
            top: root.showTitleLabel ? fieldIconItem.bottom : parent.top
            left: parent.left
            right: iconItem.left
            bottom: parent.bottom
        }

        visible: root.showTitleLabel ? root.expanded : true

        Repeater {
            id: detailsRepeater

            anchors.top: parent.top
            model: detailModel
        }
    }

    IconButton {
        id: iconItem

        anchors {
            top: root.showTitleLabel ? undefined : fieldIconItem.top
            topMargin: ContactsUtil.__editorIconsVerticalPadding
            right: parent.right
            rightMargin: Theme.paddingLarge
            verticalCenter: root.showTitleLabel ? fieldIconItem.verticalCenter : undefined
        }

        width: Theme.iconSizeSmallPlus
        height: Theme.iconSizeSmallPlus
        visible: root.showExpandButton
        icon.source: root.expanded ? "image://theme/icon-splus-up" : "image://theme/icon-splus-down"

        onClicked: root.expanded = !root.expanded
    }

    Component.onDestruction: {
        expanded = false
    }
}
