/*
 * SPDX-FileCopyrightText: 2023-2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.6
import Sailfish.Silica 1.0
import Sailfish.Contacts 1.0
import org.nemomobile.contacts 1.0

MultiTypeFieldEditor {
    id: root

    function aboutToSave() {
        if (detailModel.userModified) {
            // Copy modified values from detailEditors to detailModel before saving.
            for (var i = 0; i < detailModel.count; ++i) {
                var delegate = detailEditors.itemAt(i).addressFields

                if (!!delegate) {
                    var val = delegate.editedAddressAsString()

                    if (val) detailModel.setProperty(i, "value", val)
                }
            }

            detailModel.copyMultiTypeDetailChanges(contact, propertyAccessor)
        }
    }

    function _testHasAddressContent() {
        for (var i = 0; i < detailEditors.count; ++i) {
            var delegate = detailEditors.itemAt(i)
            if (delegate && testHasContent(delegate.addressFields)) {
                return true
            }
        }
        return false
    }

    //: Add an address for this contact
    //% "Add address"
    fieldAdditionText: qsTrId("contacts-bt-contact_add_address")
    titleIconSource: "image://theme/icon-splus-location"
    titleLabelText: ContactsUtil.getNameForDetailType(Person.AddressType)
    showTitleLabel: true

    propertyAccessor: 'addressDetails'
    valueField: 'address'
    allowedTypes: [ Person.AddressType ]
    subTypesExclusive: false
    allowedSubTypes: {
        var subTypes = {}
        subTypes[Person.AddressType] = [
                    Person.AddressSubTypeParcel,
                    Person.AddressSubTypePostal,
                    Person.AddressSubTypeDomestic,
                    Person.AddressSubTypeInternational
                ]
        return subTypes
    }

    // Each address delegate contains a set of input fields to edit each part of the address
    // (street, city etc.) individually.
    fieldDelegate: Item {
        id: addressDelegate

        property var addressFields: AddressFieldModel {}
        property int addressIndex: model.index
        property int addressSubType: model.subType === undefined ? -1 : model.subType
        property int addressLabel: model.label
        property bool aboutToDelete

        function forceActiveFocus() {
            var firstEditorField = addressFieldsRepeater.itemAt(0)
            if (firstEditorField) {
                firstEditorField.forceActiveFocus()
            }
        }

        width: parent.width
        height: addressFieldsColumn.height
                + (root.hasContent
                   ? addAddressButton.height + addAddressButton.anchors.topMargin
                   : 0)
                + (addressIndex !== 0
                   ? fieldIconItem.height + fieldIconItem.anchors.topMargin
                   : 0)

        Component.onCompleted: {
            addressFields.reload(model.value)

            // Now the address is loaded, break the button binding so that the button state doesn't
            // change whenever the model value changes.
            addAddressButton.offscreen = addAddressButton.offscreen
            addressDelegate.forceActiveFocus(addAddressButton.animationDuration)
        }

        Item {
            id: fieldIconItem

            anchors {
                top: parent.top
                left: parent.left
                leftMargin: Theme.paddingLarge
            }
            width: Theme.iconSizeSmallPlus
            height: Theme.dp(64)
            opacity: root.showTitleLabel ? 1.0 : 0.0
            visible: addressIndex !== 0

            Icon {
                id: fieldIcon

                anchors {
                    left: parent.left
                    verticalCenter: parent.verticalCenter
                }
                highlighted: root.expanded || root.highlighted
                source: root.titleIconSource
            }
        }

        Label {
            id: titleLabel

            anchors {
                verticalCenter: fieldIconItem.verticalCenter
                left: fieldIconItem.right
                leftMargin: Theme.paddingLarge
                right: clearButton.left
                rightMargin: Theme.paddingSmall
            }

            visible: root.showTitleLabel && addressIndex !== 0
            highlighted: root.expanded || root.highlighted

            text: root.titleLabelText
        }

        IconButton {
            id: clearButton

            anchors {
                left: parent.right
                verticalCenter: titleLabel.verticalCenter
            }

            visible: addressIndex !== 0
            width: Theme.iconSizeSmallPlus
            height: Theme.iconSizeSmallPlus
            icon.source: "image://theme/icon-splus-remove"
            highlighted: true

            onClicked: {
                root.detailModel.userModified = true
                if (!root.animateAndRemove(addressIndex, addressDelegate)) {
                    root.detailModel.setProperty(addressIndex, "value", root.detailModel.emptyValue)
                }
            }
        }

        Column {
            id: addressFieldsColumn

            anchors {
                top: addressIndex === 0 ? parent.top : fieldIconItem.bottom
                left: parent.left
                right: parent.right
            }

            opacity: root.expanded ? 1.0 : 0.0
            enabled: root.expanded
            spacing: 0
            Behavior on opacity {
                FadeAnimator {
                    duration: addAddressButton.animationDuration
                }
            }
            Behavior on height {
                NumberAnimation {
                    duration: addAddressButton.animationDuration
                    easing.type: Easing.InOutQuad
                }
            }

            Item {
                width: parent.width
                height: Theme.paddingSmall
            }

            Item {
                id: addressButtons

                readonly property int maxComboBoxesWidth: addressDelegate.width
                                                              - Theme.paddingLarge * 2
                                                              - Theme.iconSizeSmallPlus

                anchors {
                    left: parent.left
                    leftMargin: Theme.iconSizeSmallPlus + Theme.paddingLarge
                    right: parent.right
                    rightMargin: -Theme.iconSizeSmallPlus
                }
                height: Math.max(detailSubTypeCombo.height, detailLabelCombo.height)

                MiniComboBox {
                    id: detailSubTypeCombo

                    anchors {
                        left: parent.left
                        leftMargin: Theme.paddingLarge
                    }

                    width: ContactsUtil.calcMiniComboBoxWidth(implicitWidth,
                                                              detailLabelCombo.implicitWidth,
                                                              addressButtons.maxComboBoxesWidth)
                    menu: DetailSubTypeMenu {
                        id: detailSubTypeMenu

                        model: root.detailSubTypeModel
                        currentSubType: addressDelegate.addressSubType

                        onCurrentIndexChanged: detailSubTypeCombo.currentIndex = currentIndex
                        onSubTypeClicked: root.setDetailType(addressDelegate.addressIndex, type, subType)
                    }
                }

                MiniComboBox {
                    id: detailLabelCombo

                    anchors {
                        left: detailSubTypeCombo.right
                    }

                    width: ContactsUtil.calcMiniComboBoxWidth(implicitWidth,
                                                              detailSubTypeCombo.implicitWidth,
                                                              addressButtons.maxComboBoxesWidth)

                    value: (currentItem == null || currentIndex === 0)
                           ? noSelectLabel.text
                           : currentItem.text

                    menu: DetailLabelMenu {
                        id: detailLabelMenu

                        currentLabel: addressDelegate.addressLabel

                        onCurrentIndexChanged: detailLabelCombo.currentIndex = currentIndex
                        onLabelClicked: root.setDetailLabel(addressDelegate.addressIndex, label)
                    }

                    onClicked: focus = true

                    // Dummy to get the label width calculation (cannot use TextMetrics::width, does not update binding as expected)
                    Label {
                        id: noSelectLabel
                        visible: false
                        text: ContactsUtil.getSelectLabelText()
                    }
                }
            }

            Item {
                width: parent.width
                height: Theme.paddingMedium
            }

            // Repeater of individual address fields (street, city etc.)
            Repeater {
                id: addressFieldsRepeater

                model: addressDelegate.addressFields

                delegate: EditorFieldDelegate {
                    detailType: Person.AddressType
                    leftMargin: 0
                    buttonMode: false
                    editor: root
                    focus: root.initialFocusIndex === addressDelegate.addressIndex
                           && addressIndex === 0 // always focus the first address field

                    onModified: {
                        if (addressDelegate.aboutToDelete) {
                            return
                        }

                        var trimmedValue = value.trim()
                        addressFields.setProperty(index, "value", trimmedValue)
                        root.resetField(addressDelegate.addressIndex)
                        root.hasContent = trimmedValue.length > 0 || root._testHasAddressContent()

                        addAddressButton.checkVisibility()
                    }
                }
            }
        }

        AddFieldButton {
            id: addAddressButton

            property bool _visible

            function checkVisibility() {
                _visible = root.hasContent
                        && !addressDelegate.addressFields.allFieldsEmpty()
                        && model.index === root.detailModel.count - 1
            }

            anchors {
                top: addressFieldsColumn.bottom
                left: parent.left
                leftMargin: Theme.iconSizeSmallPlus + Theme.paddingLarge * 2
                right: parent.right
            }

            height: _visible ? implicitHeight : anchors.topMargin

            offscreen: !addressDelegate.addressFields.allFieldsEmpty()

            opacity: _visible ? 1.0 : 0.0
            text: root.fieldAdditionText
            showIconWhenOffscreen: false
            animate: root.ready
            highlighted: down

            onClicked: {
                _visible = false
                root.addEmptyField()
            }

            onEnteredButtonMode: {
                root.resetField(addressFieldsRepeater.model.index)
            }
        }

        Connections {
            target: root.detailModel
            onCountChanged: {
                addAddressButton.checkVisibility()
            }
        }
    }
}
