/*
 * SPDX-FileCopyrightText: 2023-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.6
import Sailfish.Silica 1.0

Item {
    id: root

    property alias text: buttonLabel.text
    property alias icon: iconContainer

    property bool offscreen
    readonly property real offscreenPeekWidth: Theme.paddingMedium + iconContainer.width + Theme.paddingSmall
    property bool showIconWhenOffscreen: true

    property bool animate
    property int animationDuration: 350 // Same as removal animation in BaseEditor
    readonly property real revealedContentOpacity: _revealedContentOpacity
    readonly property bool busy: toButtonState.running || toDefaultState.running
    property bool highlighted: down
    readonly property bool down: mouseArea.pressed

    property real _revealedContentOpacity

    signal clicked()
    signal enteredButtonMode()
    signal exitedButtonMode()

    implicitHeight: Theme.iconSizeMedium
    height: implicitHeight

    Label {
        id: buttonLabel

        anchors {
            left: parent.left
            right: iconContainer.left
            verticalCenter: parent.verticalCenter
        }

        font.pixelSize: Theme.fontSizeMedium
        color: root.highlighted ? Theme.highlightColor : Theme.primaryColor
        truncationMode: TruncationMode.Fade
        horizontalAlignment: Text.AlignLeft
    }

    HighlightImage {
        id: iconContainer

        anchors {
            left: parent.right
            verticalCenter: parent.verticalCenter
        }

        width: Theme.iconSizeSmallPlus
        height: Theme.iconSizeSmallPlus
        highlighted: root.highlighted
        source: "image://theme/icon-splus-new"
    }

    // Close vkb if user clicks within the button area but outside of text+icon.
    MouseArea {
        id: mouseArea

        anchors {
            top: parent.top
            bottom: parent.bottom
            left: parent.left
            right: iconContainer.right
        }

        enabled: root.state === "" && !root.busy
        hoverEnabled: true
        cursorShape: enabled ? Qt.PointingHandCursor : Qt.ArrowCursor

        onPressed: root.focus = true
        onClicked: root.clicked()
    }

    states: [
        State {
            name: "offscreen"
            when: root.offscreen

            PropertyChanges {
                target: root
                x: -root.width + root.offscreenPeekWidth
            }
            PropertyChanges {
                target: buttonLabel
                opacity: 0
            }
            PropertyChanges {
                target: iconContainer
                opacity: showIconWhenOffscreen ? 1 : 0
            }
            PropertyChanges {
                target: root
                _revealedContentOpacity: 1
            }
        }
    ]

    transitions: [
        Transition {
            id: toDefaultState

            from: ""; to: "offscreen"
            enabled: root.animate

            SequentialAnimation {
                PauseAnimation {
                    duration: root.animationDuration / 2
                }
                FadeAnimation {
                    target: root
                    property: "_revealedContentOpacity"
                    duration: root.animationDuration
                }
            }
            NumberAnimation {
                target: root
                property: "x"
                duration: root.animationDuration
                easing.type: Easing.InOutQuad
            }
            FadeAnimation {
                targets: [buttonLabel, iconContainer]
                duration: root.animationDuration / 2
            }
            ScriptAction {
                script: root.exitedButtonMode()
            }
        },
        Transition {
            id: toButtonState

            from: "offscreen"; to: ""
            enabled: root.animate

            SequentialAnimation {
                ParallelAnimation {
                    FadeAnimation {
                        target: root
                        property: "_revealedContentOpacity"
                        duration: root.animationDuration / 2
                    }
                    NumberAnimation {
                        target: root
                        property: "x"
                        duration: root.animationDuration
                        easing.type: Easing.InOutQuad
                    }
                    FadeAnimation {
                        target: buttonLabel
                        duration: root.animationDuration
                    }
                }
                ScriptAction {
                    script: root.enteredButtonMode()
                }
            }
        }
    ]
}
