/**
 * SPDX-FileCopyrightText: 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.6
import Aurora.Controls 1.0
import Sailfish.Silica 1.0
import Sailfish.Contacts 1.0 as SailfishContacts
import org.nemomobile.contacts 1.0
import org.nemomobile.commhistory 1.0
import Nemo.DBus 2.0

ContactCardSection {
    id: root

    header: qsTrId("components_contacts-la-activity")

    enabled: repeater.count > 0

    property QtObject _messagesInterface

    property Person _contact: contact
    property bool _hidePhoneActions: hidePhoneActions
    property var _simManager: SailfishContacts.ContactCardUtils.simManager

    function startPhoneCall(number, modemPath) {
        if (modemPath !== undefined && modemPath !== "") {
            voicecall.dialViaModem(modemPath, number)
        } else {
            voicecall.dial(number)
        }
    }

    function messagesInterface() {
        if (!_messagesInterface) {
            _messagesInterface = Qt.createQmlObject('import "../common"; MessagesInterface { }', root)
            if (!_messagesInterface)
                console.warn("ContactCardPage: Failed creating MessagesInterface instance")
        }
        return _messagesInterface
    }

    function startSms(number) {
        messagesInterface().startSMS(number)
    }

    function startInstantMessage(localUid, remoteUid) {
        messagesInterface().startConversation(localUid, remoteUid)
    }

    fields: Repeater {
        id: repeater

        model: contactEventModel

        ContactEventField {
            visible: index === 0

            width: parent.width

            onStartPhoneCall: root.startPhoneCall(number, modemPath)
            onStartSms: root.startSms(number)
            onStartInstantMessage: root.startInstantMessage(localUid, remoteUid)
        }
    }

    Item {
        parent: root

        width: parent.width
        height: Theme.dp(80)

        Label {
            anchors.left: parent.left
            anchors.right: rightIcon.left
            anchors.margins: Theme.paddingMedium
            height: parent.height

            font.pixelSize: Theme.fontSizeSmall
            truncationMode: TruncationMode.Fade
            verticalAlignment: Qt.AlignVCenter

            color: seeAllMouseArea.pressed ? Theme.highlightColor : Theme.primaryColor

            //% "See all activity"
            text: qsTrId("components_contacts-la-see_all_activity")
        }

        HighlightImage {
            id: rightIcon

            width: Theme.dp(48)
            height: width

            highlighted: seeAllMouseArea.pressed

            anchors {
                right: parent.right
                rightMargin: Theme.paddingMedium
                verticalCenter: parent.verticalCenter
            }
            
            source: "image://theme/icon-splus-right"
        }

        MouseArea {
            id: seeAllMouseArea

            anchors.fill: parent
            cursorShape: Qt.PointingHandCursor

            onClicked: pageStack.animatorPush(activityPageComponent)
        }

        Component {
            id: activityPageComponent

            SailfishContacts.ContactActivityPage {
                hidePhoneActions: root._hidePhoneActions
                contact: root._contact
                simManager: SailfishContacts.ContactCardUtils.simManager

                onStartPhoneCall: root.startPhoneCall(number, modemPath)
                onStartSms: root.startSms(number)
                onStartInstantMessage: root.startInstantMessage(localUid, remoteUid)
            }
        }
    }

    CommRecipientEventModel {
        id: contactEventModel

        contactId: contact ? contact.id : 0
        remoteUid: contact && contact.id === 0
                   ? SailfishContacts.ContactsUtil.firstPhoneNumber(contact)
                   : ""
        limit: 1
    }

    DBusInterface {
        id: voicecall

        service: "com.jolla.voicecall.ui"
        path: "/"
        iface: "com.jolla.voicecall.ui"

        function dial(number) {
            call('dial', number)
        }

        function dialViaModem(modemPath, number) {
            call('dialViaModem', [ modemPath, number ])
        }
    }
}
