/**
 * SPDX-FileCopyrightText: 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.6
import Aurora.Controls 1.0
import Sailfish.Silica 1.0
import Sailfish.Policy 1.0
import Sailfish.Telephony 1.0
import Sailfish.Contacts 1.0 as SailfishContacts
import org.nemomobile.contacts 1.0
import org.nemomobile.commhistory 1.0
import org.nemomobile.configuration 1.0
import Aurora.Applications.Internal 1.0 as AppInternal
import Nemo.DBus 2.0

MouseArea {
    id: root

    readonly property bool highlighted: pressed || contextMenu.instance.opened

    readonly property bool isPhone: model.localUid.indexOf('/ring/tel/') >= 0
    readonly property bool isMessage: model.eventType !== CommCallModel.CallEvent
                                      && model.eventType !== CommCallModel.VoicemailEvent

    readonly property bool showYear: model.endTime.getFullYear() !== Date().getFullYear()

    readonly property int timeFormatType: timeFormatConfig.value === "12"
                                           ? Formatter.TimeValueTwelveHours
                                           : Formatter.TimeValueTwentyFourHours

    readonly property SimManager simManager: SailfishContacts.ContactCardUtils.simManager

    readonly property string imsi: model.subscriberIdentity

    readonly property int modemIndex: simManager && simManager.simNames.length && imsi.length > 0
                                      ? simManager.indexOfModemFromImsi(imsi)
                                      : -1

    signal startPhoneCall(string number, string modemPath)
    signal startSms(string number)
    signal startInstantMessage(string localUid, string remoteUid)

    function getActionTitle() {
        if (model.direction === CommCallModel.Inbound) {
            return model.isMissedCall
            //% "Missed call"
                    ? qsTrId("components_contacts-la-missed_call")
                      //% "Incoming call"
                    : qsTrId("components_contacts-la-incoming_call")
        } else {
            //% "Outgoing call"
            return qsTrId("components_contacts-la-outgoing_call")
        }
    }

    function tryOpenContextMenu() {
        if (!isPhone || !hidePhoneActions) {
            contextMenu.open()
        }
    }

    implicitHeight: Theme.dp(103)
    cursorShape: Qt.PointingHandCursor
    acceptedButtons: Qt.LeftButton | Qt.RightButton

    onClicked: root.tryOpenContextMenu()
    onPressAndHold: root.tryOpenContextMenu()

    ConfigurationValue {
        id: timeFormatConfig

        key: "/sailfish/i18n/lc_timeformat24h"
    }

    DBusInterface {
        id: voicecall

        service: "com.jolla.voicecall.ui"
        path: "/"
        iface: "com.jolla.voicecall.ui"

        function dial(number) {
            call('dial', number)
        }

        function dialViaModem(modemPath, number) {
            call('dialViaModem', [ modemPath, number ])
        }
    }

    AppInternal.LazyPopupMenu {
        id: contextMenu

        sourceComponent: PopupMenu {
            id: _contextMenu

            property alias _contentColumn: _contextMenu

            PopupMenuItem {
                visible: root.isPhone
                //% "Call"
                text: qsTrId("components_contacts-me-call")
                icon.source: "image://theme/icon-splus-call"

                onClicked: {
                    if (Telephony.voiceSimUsageMode === Telephony.AlwaysAskSim &&
                            root.simManager.availableModemCount > 0) {
                        simMenu.open()
                    } else {
                        root.startPhoneCall(model.remoteUid, "")
                    }
                }
            }

            PopupMenuItem {
                //% "Send message"
                text: qsTrId("components_contacts-me-send_message")
                opacity: AccessPolicy.smsEnabled ? 1 : Theme.opacityHigh
                icon.source: "image://theme/icon-splus-message"

                onClicked: {
                    if (root.isPhone) {
                        if (AccessPolicy.smsEnabled) {
                            root.startSms(model.remoteUid)
                        } else {
                            SailfishContacts.ContactsUtil.showSmsNotice()
                        }
                    } else {
                        root.startInstantMessage(model.localUid, model.remoteUid)
                    }
                }
            }

            Component.onCompleted: {
                _contextMenu.preferredWidth = AppInternal.UiUtils.getPopupMenuPreferredWidth({
                    popupMenu: _contextMenu,
                    minimumWidth: AppInternal.UiUtils.isLargeScreen
                                    ? _contextMenu.preferredWidth
                                    : root.width,
                    maximumWidth: root.width
                })
            }
        }
    }

    AppInternal.LazyPopupMenu {
        id: simMenu

        sourceComponent: PopupMenu {
            id: _simPopupMenu

            preferredWidth: contextMenu.instance.preferredWidth

            SimPicker {
                id: simPicker

                onSimSelected: {
                    root.startPhoneCall(model.remoteUid, modemPath)

                    _simPopupMenu.close()
                }
            }
        }
    }

    Item {
        id: icon

        width: Theme.dp(32)
        height: width

        HighlightImage {
            id: mainImage

            anchors.fill: parent
            highlighted: root.highlighted

            source: {
                if (isMessage) {
                    return "image://theme/icon-s-message"
                } else if (model.eventType === CommCallModel.VoicemailEvent) {
                    return "image://theme/icon-m-voicemail"
                } else {
                    if (model.direction === CommCallModel.Inbound) {
                        return model.isMissedCall
                                ? "image://theme/icon-s-activity-missed-call"
                                : "image://theme/icon-s-activity-incoming-call"
                    } else {
                        return "image://theme/icon-s-activity-outgoing-call"
                    }
                }
            }
        }
    }

    Column {
        id: content

        anchors {
            left: icon.right
            leftMargin: Theme.paddingLarge
            right: timeStamp.left
            rightMargin: Theme.paddingSmall
        }

        spacing: Theme.paddingSmall

        Label {
            width: parent.width
            font.pixelSize: Theme.fontSizeMedium
            color: root.highlighted ? Theme.highlightColor : Theme.primaryColor
            textFormat: Text.PlainText
            truncationMode: Text.ElideRight
            wrapMode: Text.Wrap
            maximumLineCount: 1
            text: {
                if (isMessage) {
                    var messageText
                    if (freeText) {
                        messageText = freeText
                    } else if (subject) {
                        messageText = subject
                    } else if (model.eventType === CommHistory.MMSEvent) {
                        //% "Multimedia message"
                        messageText = qsTrId("components_contacts-la-multimedia_message")
                    } else {
                        //% "Message"
                        messageText = qsTrId("components_contacts-la-message")
                    }
                    return messageText
                } else {
                    return getActionTitle()
                }
            }
        }

        Label {
            width: parent.width
            text: AsYouTypeFormatterUtil.getFormattedPhoneNumber(model.remoteUid)
            font.pixelSize: Theme.fontSizeExtraSmall
            color: root.highlighted ? Theme.secondaryHighlightColor : Theme.secondaryColor
            truncationMode: TruncationMode.Fade
        }

        Row {
            height: Theme.dp(32)
            width: parent.width
            spacing: Theme.paddingMedium

            Row {
                visible: !root.isMessage
                height: parent.height
                spacing: Theme.paddingSmall

                HighlightImage {
                    width: height
                    height: parent.height
                    highlighted: root.highlighted
                    source: "image://theme/icon-s-duration"
                }

                Label {
                    id: durationLabel

                    property int duration: (model.endTime.valueOf() - model.startTime.valueOf()) / 1000

                    height: parent.height
                    text: Format.formatDuration(duration, duration >= 3600 ? Formatter.DurationLong : Formatter.DurationShort)
                    font.pixelSize: Theme.dp(25)
                    color: root.highlighted ? Theme.secondaryHighlightColor : Theme.secondaryColor
                    truncationMode: TruncationMode.Fade
                }
            }

            Row {
                height: parent.height
                spacing: Theme.paddingSmall
                visible: simManager && simManager.enabledModems.length > 1

                HighlightImage {
                    width: height
                    height: parent.height
                    highlighted: root.highlighted
                    source: {
                        switch (root.modemIndex) {
                            case 0: return "image://theme/icon-s-sim-1"
                            case 1: return "image://theme/icon-s-sim-2"
                            default: return ""
                        }
                    }
                }

                Label {
                    text: root.modemIndex >= 0
                          ? root.simManager.modemSimModel.get(root.modemIndex).operatorDescription
                          : ""
                    truncationMode: TruncationMode.Elide
                    font.pixelSize: Theme.dp(25)
                    color: root.highlighted ? Theme.secondaryHighlightColor : Theme.secondaryColor
                }
            }
        }
    }

    Column {
        id: timeStamp

        width: childrenRect.width
        anchors.right: parent.right
        spacing: Theme.dp(4)
        topPadding: Theme.dp(4)

        Label {
            height: Theme.dp(32)
            anchors.right: parent.right
            text: Format.formatDate(model.endTime, root.timeFormatType)
            font.pixelSize: Theme.fontSizeExtraSmall
            color: root.highlighted ? Theme.secondaryHighlightColor : Theme.secondaryColor
            truncationMode: TruncationMode.Fade
            horizontalAlignment: Qt.AlignRight
        }

        Label {
            anchors.right: parent.right
            text: Format.formatDate(model.endTime, showYear ? Format.DateMedium : Format.DateMediumWithoutYear)
            font.pixelSize: Theme.fontSizeExtraSmall
            color: root.highlighted ? Theme.secondaryHighlightColor : Theme.secondaryColor
            horizontalAlignment: Qt.AlignRight
        }
    }
}
