/**
 * SPDX-FileCopyrightText: 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.0
import Sailfish.Silica 1.0

MouseArea {
    id: root

    readonly property real componentHeight: Theme.dp(100)
    readonly property bool highlighted: pressed ||
                                        (detailContextMenu.opened
                                         && detailContextMenu.detailsIndex === detailsIndex
                                         && detailContextMenu.detailsType === detailsType)
    property bool isNoteSection
    property alias mainText: mainLabel.text
    property alias secondText: secondLabel.text
    property alias mainIcon: mainImage.source
    property alias secondIcon: secondImage.source
    property alias secontIconEnabled: secondIconItem.enabled
    property bool actionsEnabled: true
    property bool isContactStorageSection
    property bool turnOffPolicy: true

    signal contextMenuRequested()
    signal secondIconClicked()

    implicitHeight: {
        if (root.isNoteSection) {
            var noteLabelHeight = mainLabel.y + mainLabel.height + secondLabel.height
            return noteLabelHeight > root.componentHeight
                    ? noteLabelHeight
                    : root.componentHeight
        }

        return root.componentHeight
    }

    acceptedButtons: Qt.LeftButton | Qt.RightButton
    cursorShape: Qt.PointingHandCursor

    onPressAndHold: contextMenuRequested()

    onClicked: {
        if (mouse.button === Qt.RightButton) {
            contextMenuRequested()
        }
    }

    Item {
        id: mainIconItem

        anchors.verticalCenter: isContactStorageSection ? parent.verticalCenter : undefined
        y: isContactStorageSection ? 0 : Theme.dp(15)
        width: Theme.dp(48)
        height: width
        enabled: root.actionsEnabled

        Image {
            id: mainImage

            anchors.fill: parent
            opacity: root.actionsEnabled ? 1.0 : Theme.opacityLow
        }
    }

    Label {
        id: mainLabel

        anchors {
            left: mainIconItem.right
            leftMargin: Theme.paddingLarge
            right: secondIconItem.left
            verticalCenter: isContactStorageSection ? mainIconItem.verticalCenter : undefined
        }

        y: Theme.paddingMedium
        font.pixelSize: Theme.fontSizeMedium
        color: root.highlighted ? Theme.highlightColor : Theme.primaryColor
        truncationMode: TruncationMode.Fade
        opacity: mainImage.opacity
    }

    Label {
        id: secondLabel

        anchors {
            top: mainLabel.bottom
            topMargin: Theme.paddingMedium
            left: mainLabel.left
            right: mainLabel.right
        }

        font.pixelSize: Theme.fontSizeExtraSmall
        color: root.highlighted ? Theme.secondaryHighlightColor : Theme.secondaryColor
        truncationMode: TruncationMode.Fade
        opacity: mainImage.opacity
    }

    Item {
        id: secondIconItem

        anchors.right: parent.right
        y: Theme.dp(15)
        width: Theme.dp(48)
        height: width

        Image {
            id: secondImage

            opacity: root.actionsEnabled && root.turnOffPolicy ? 1.0 : Theme.opacityLow
            anchors.fill: parent
        }

        MouseArea {
            anchors.fill: parent
            visible: root.actionsEnabled && secondImage.status === Image.Ready
            cursorShape: Qt.PointingHandCursor
            enabled: (root.actionsEnabled && secondImage.status === Image.Ready)

            onClicked: root.secondIconClicked()
        }
    }
}
