/**
 * SPDX-FileCopyrightText: Copyright 2023-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.6
import Sailfish.Silica 1.0
import Sailfish.Policy 1.0
import Aurora.Controls 1.0
import Sailfish.Contacts 1.0
import Sailfish.Telephony 1.0
import org.nemomobile.contacts 1.0

PopupMenu {
    id: root

    property Item delegate: root.parent
    property var person: null
    property bool quickCall: mainPage.quickCall
    property bool multiselect

    readonly property bool _hasPhoneNumber: root.person
                                            ? root.person.phoneDetails.length > 0
                                            : false

    function openContactCard(person, remoteUid) {
        var personObject = !!person ? person : ContactCreator.createContact({"phoneNumbers": [remoteUid]})
        pageStack.animatorPush("Sailfish.Contacts.ContactCardPage",
                               { "contact": personObject, "activeDetail": remoteUid })
    }

    function dial(remoteUid, modemPath) {
        telephony.dial(remoteUid, modemPath)
        call()
    }

    preferredWidth: Screen.sizeCategory > Screen.Medium
                    ?  Math.max(Screen.width, Screen.height) * 0.33333333 - Theme.horizontalPageMargin
                    :  Math.min(Screen.width, Screen.height) - Theme.horizontalPageMargin * 2

    onOpenedChanged: {
        if (!root.delegate) {
            return
        }

        if (root.opened) {
            root.delegate.highlighted = root.delegate.down || root.opened
        } else {
            root.delegate.highlighted = Qt.binding(function() { return this.down })
        }
    }

    PopupMenuItem {
        //% "Call"
        text: qsTrId("components_contacts-me-call")
        icon.source: "image://theme/icon-splus-call"
        visible: !root.quickCall && root._hasPhoneNumber
                 && telephony.callingPermitted && root.person.phoneDetails.length === 1
        onClicked: dial(root.person.phoneDetails[0].number)
    }

    PropertyPopupSelector {
        //% "Call"
        text: qsTrId("components_contacts-me-call")
        icon.source: "image://theme/icon-splus-call"
        visible: !root.quickCall && root._hasPhoneNumber
                 && telephony.callingPermitted && root.person.phoneDetails.length > 1
        requiredProperty: PeopleModel.PhoneNumberRequired
        contact: root.person

        onPropertySelected: {
            dial(propertyData.property.number)
        }
    }

    PopupMenuItem {
        //% "Open contact card"
        text: qsTrId("components_contacts-me-open_contact_card")
        icon.source: "image://theme/icon-splus-contact"
        visible: root.quickCall && root._hasPhoneNumber
        onClicked: {
            if (!root.person) {
                return
            }
            openContactCard(root.person, root.person.phoneDetails[0].number)
        }
    }

    PopupMenuItem {
        //% "Send message"
        text: qsTrId("components_contacts-me-send_message")
        icon.source: "image://theme/icon-splus-message"
        visible: root._hasPhoneNumber && telephony.messagingPermitted && root.person.phoneDetails.length === 1
        opacity: AccessPolicy.smsEnabled ? 1 : Theme.opacityHigh

        onClicked: {
            if (AccessPolicy.smsEnabled) {
                messaging.startSMS(root.person.phoneDetails[0].number)
            } else {
                ContactsUtil.showSmsNotice()
            }
        }
    }

    PropertyPopupSelector {
        //% "Send message"
        text: qsTrId("components_contacts-me-send_message")
        icon.source: "image://theme/icon-splus-message"
        visible: root._hasPhoneNumber && telephony.messagingPermitted && root.person.phoneDetails.length > 1
        requiredProperty: PeopleModel.PhoneNumberRequired
        contact: root.person
        opacity: AccessPolicy.smsEnabled ? 1 : Theme.opacityHigh

        onClicked: {
            if (!AccessPolicy.smsEnabled) {
                ContactsUtil.showSmsNotice()
                close()
            }
        }

        onPropertySelected: {
                messaging.startSMS(propertyData.property.number)
        }
    }

    PopupMenuItem {
        text: {
            if (!visible) {
                return ""
            }

            if (ContactModelCache.isBlockedContact(root.person)) {
                //% "Unblock contact"
                return qsTrId("components_contacts-me-unblock_contact")
            }

            //% "Block contact"
            return qsTrId("components_contacts-me-block_contact")
        }

        visible: root._hasPhoneNumber
        icon.source: "image://theme/icon-splus-blocked"
        onClicked: {
            if (!root.person) {
                return
            }

            const blockedName = root.person
                              ? root.person.primaryName + " " + root.person.secondaryName
                              : root.person.phoneDetails

            const editorProperties = {
                "blockedName": blockedName,
                "contact": root.person
            }

            pageStack.animatorPush("Sailfish.Contacts.BlockContactDialog", editorProperties, PageStackAction.Animated)
        }
    }

    PopupMenuItem {
        readonly property bool isDeletable: !!root.delegate
                                            && !!root.delegate.canDeleteContact
                                            && ContactsUtil.allConstituentsDeletable(root.person,
                                                                                     ContactModelCache.unfilteredModel())

        //% "Delete contact"
        text: qsTrId("components_contacts-me-delete_contact")
        icon.source: "image://theme/icon-splus-delete"
        opacity: isDeletable ? 1.0 : Theme.opacityHigh

        onClicked: {
            if (!root.person) {
                return
            }

            if (!isDeletable) {
                const notice = ContactsUtil.contactUnableActionsNames[ContactsUtil.ActionsWithConstContact.Delete]
                ContactsUtil.showNotice(notice,
                                        Notice.Short,
                                        Notice.Bottom,
                                        0,
                                        -Theme.dp(142))
                return
            }

            var obj = pageStack.animatorPush("Sailfish.Contacts.RemoveContactsDialog",
                                             {
                                                 "person": root.person
                                             })
            obj.pageCompleted.connect(function(page) {
                page.accepted.connect(function() {
                    ContactModelCache.deleteContact(root.person)
                })
            })
        }
    }
}
