/*
 * SPDX-FileCopyrightText: 2024-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.6
import Aurora.Controls 1.0
import Sailfish.Silica 1.0
import Sailfish.Contacts 1.0
import org.nemomobile.contacts 1.0

PopupMenu {
    id: contextMenu

    property var person
    property Item delegate: contextMenu.parent
    property bool multiselect: true
    property PeopleModel allContactsModel: PeopleModel {
        filterType: PeopleModel.FilterAll
        requiredProperty: PeopleModel.NoPropertyRequired
    }

    readonly property bool isEditable: ContactsUtil.isWritableContact(person.constituents.length > 1
                                                                      ? person
                                                                      : allContactsModel.personById(person.constituents[0]))
    readonly property bool isDeletable: !!person
                                        && person.id !== 0
                                        && ContactsUtil.allConstituentsDeletable(person, allContactsModel)

    signal _parentDestroyed()
    signal changeFavorites(var person, bool status)
    signal openMultiSelectPage()
    signal deleteContact(var person)

    preferredWidth: Screen.sizeCategory > Screen.Medium
                    ?  Math.max(Screen.width, Screen.height) * 0.33333333 - Theme.horizontalPageMargin
                    :  Math.min(Screen.width, Screen.height) - Theme.horizontalPageMargin * 2
    preferredHeight: Screen.height

    onOpenedChanged: {
        if (contextMenu.opened) {
            contextMenu.delegate.highlighted = contextMenu.delegate.down || contextMenu.opened
        } else {
            contextMenu.delegate.highlighted = Qt.binding(function() { return this.down })
        }
    }

    onPersonChanged: {
        if (person)
            person.fetchConstituents()
    }

    PopupMenuItem {
        opacity: contextMenu.isEditable ? 1.0 : Theme.opacityHigh
        icon.source: "image://theme/icon-splus-edit"
        //: Edit contact, from list
        //% "Edit"
        text: qsTrId("components_contacts-me-edit_contact")

        onClicked: {
            if (!contextMenu.isEditable) {
                ContactsUtil.showNotice(ContactsUtil.contactUnableActionsNames[ContactsUtil.ActionsWithConstContact.Edit],
                                                         Notice.Short,
                                                         Notice.Bottom,
                                                         0,
                                                         -Theme.dp(142))
                return
            }

            ContactsUtil.editContact(contextMenu.person, contextMenu.allContactsModel, pageStack)
        }
    }

    PopupMenuItem {
        enabled: !ContactModelCache.isBlockedContact(contextMenu.person)
        icon.source: contextMenu.person && contextMenu.person.favorite
                     ? "image://theme/icon-splus-favorite-selected"
                     : "image://theme/icon-splus-favorite"
        text: contextMenu.person && contextMenu.person.favorite
        //: Set contact as not favoriteopenContactMultiSelectPage
        //% "Remove from favorites"
              ? qsTrId("components_contacts-me-remove_contact_from_favorites")
                //: Set contact as favorite
                //% "Add to favorites"
              : qsTrId("components_contacts-me-add_contact_to_favorites")
        visible: !ContactModelCache.isBlockedContact(contextMenu.person)

        onClicked: {
            changeFavorites([contextMenu.person], !contextMenu.person.favorite)
        }
    }

    PopupMenuItem {
        enabled: true
        icon.source: "image://theme/icon-splus-blocked"
        text: ContactModelCache.isBlockedContact(contextMenu.person)
        //: Unblock contact
        //% "Unblock contact"
              ? qsTrId("components_contacts-unblock_contact")
                //: Block contact
                //% "Block contact"
              : qsTrId("components_contacts-block_contact")

        onClicked: {
            pageStack.animatorPush("Sailfish.Contacts.BlockContactDialog",
                                   {
                                       "contact": contextMenu.person,
                                       "blockedName": contextMenu.person.displayLabel
                                   })
        }
    }

    PopupMenuItem {
        icon.source: "image://theme/icon-splus-delete"
        //: Delete contact
        //% "Delete"
        text: qsTrId("components_contacts-me-delete_contact")
        opacity: contextMenu.isDeletable ? 1.0 : Theme.opacityHigh

        onClicked: {
            if (!contextMenu.isDeletable) {
                const notice = ContactsUtil.contactUnableActionsNames[ContactsUtil.ActionsWithConstContact.Delete]
                ContactsUtil.showNotice(notice,
                                        Notice.Short,
                                        Notice.Bottom,
                                        0,
                                        -Theme.dp(142))
                return
            }

            if (!contextMenu.person) {
                return
            }

            deleteContact(contextMenu.person)
        }
    }

    PopupMenuItem {
        visible: contextMenu.multiselect
        icon.source: "image://theme/icon-splus-multiselect"
        //: Allows the user to select multiple contacts (for deletion or sharing)
        //% "Select contacts"
        text: qsTrId("components_contacts-me-select_contacts")
        onClicked: openMultiSelectPage()
    }
}
