/**
 * SPDX-FileCopyrightText: 2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.6
import Aurora.Controls 1.0
import Sailfish.Silica 1.0
import Sailfish.Silica.private 1.0
import Sailfish.Contacts 1.0 as SailfishContacts
import org.nemomobile.contacts 1.0

import "common/common.js" as Utils

Page {
    id: root

    property bool isContactsPopupMenu: true
    property var contactPopupMenu: null
    property string _searchPattern
    readonly property bool _searchFiltered: _searchPattern.length > 0
    readonly property var _regExp: new RegExp("^[+]*[(]{0,1}[0-9]{1,3}[)]{0,1}[-\s\.\/0-9]*$")

    readonly property bool hasContent: contactsView.count > 0

    // Left margin for ContactItem when searching
    property int contactItemLeftMargin: 0

    property PeopleModel contactsModel: PeopleModel {
        filterType: PeopleModel.FilterNone
        filterPattern: root._searchPattern
        requiredProperty: PeopleModel.NoPropertyRequired
        searchableProperty: PeopleModel.AccountUriSearchable | PeopleModel.EmailAddressSearchable | PeopleModel.PhoneNumberSearchable | PeopleModel.OrganizationSearchable
    }

    property PeopleModel allContactsModel: PeopleModel {
        filterType: PeopleModel.FilterAll
        requiredProperty: PeopleModel.NoPropertyRequired
    }

    readonly property bool isTablet: LayoutClassProvider.currentDisplayClass & DisplayLayoutClass.Tablet
    readonly property bool isPortrait: orientation & Orientation.PortraitMask
    readonly property real landscapeWidth: root.isPortrait ? root.width : Screen.width

    Component.onCompleted: {
        searchField.forceActiveFocus()
        root.contactsModel.prepareSearchFilters()
        var component
        if (isContactsPopupMenu) {
            component = Qt.createComponent(Qt.resolvedUrl("common/ContactsPopupMenu.qml"))
        } else {
            component = Qt.createComponent(Qt.resolvedUrl("common/PhonePopupMenu.qml"))
        }
        if (component.status == Component.Ready) {
            contactPopupMenu = component.createObject(root);
        }
    }

    Connections {
        target: contactPopupMenu

        onChangeFavorites: {
            favoriteModifier.setFavoriteStatus(person, status)
        }

        onOpenMultiSelectPage: {
            openContactMultiSelectPage()
        }
    }

    ContactFavoriteModifier {
        id: favoriteModifier

        peopleModel: root.allContactsModel
    }

    SailfishContacts.ContactsRemover {
        id: remover

        property int _removedCount
        property bool _error
        property bool _single

        function _statusText() {
            if (_error) {
                //: Error while removing contacts
                //% "Unable to remove contacts."
                return qsTrId("contacts-me-unable_to_remove_contacts")
            } else if (_single) {
                //% "Contact removed"
                return qsTrId("contacts-me-removed_contact")
            } else if (_removedCount > 0) {
                //% "Contacts removed (%n)"
                return qsTrId("contacts-me-removed_n_contacts", _removedCount)
            } else {
                //% "No contacts removed."
                return qsTrId("contacts-me-no_contacts_removed")
            }
        }

        onRemovingFinished: {
            _removedCount = removedCount
            Notices.show(_statusText(), Notice.Short, Notice.Bottom)
        }

        onRemovingFailed: {
            _error = true
            Notices.show(_statusText(), Notice.Short, Notice.Bottom)
        }
    }

    SailfishContacts.ContactSelectionModel {
        id: contactSelectionModel

        onSelectionAdded: {
            if (count == 1 && propertyType != undefined) {
                if (propertyType == 'accountUri' && root._isDialog) {
                    // No further selections are currently allowed - accept the dialog
                    root._page.acceptcontextMenu()
                } else if (uniformSelectionTypes) {
                    // Further selections must be of the same type
                    if (propertyType == 'phoneNumber') {
                        root._filterProperty = PeopleModel.PhoneNumberRequired
                    } else if (propertyType == 'emailAddress') {
                        root._filterProperty = PeopleModel.EmailAddressRequired
                    } else if (propertyType == 'accountUri') {
                        root._filterProperty = PeopleModel.AccountUriRequired
                    }
                }
            }
        }

        onSelectionRemoved: {
            if (count == 0 && uniformSelectionTypes) {
                root._filterProperty = root.requiredContactProperty
            }
        }
    }

    AppBar {
        id: appBar

        AppBarSearchField {
            id: searchField

            backgroundEnabled: true
            onTextChanged: {
                root._searchPattern = text
                if (text.length === 0) {
                    searchField.forceActiveFocus()
                }
            }

            //% "Search"
            placeholderText: qsTrId("components_contacts-ph-search_people")
        }

        AppBarSpacer {}
    }

    PageBusyIndicator {
        id: busyIndicator

        z: 1

        running: !root.hasContent
                 && root._searchFiltered
                 && !placeholder.enabled

        FadeAnimator {
            target: busyIndicator
            duration: 1000
            easing.type: Easing.InExpo
            to: 1.0
        }
    }

    ViewPlaceholder {
        id: placeholder

        anchors.centerIn: parent
        enabled: !root.hasContent && root._searchFiltered
        //% "No contacts found"
        text: qsTrId("components_contacts-ph-no_contacts_found")

        onEnabledChanged: {
            busyIndicator.running = false
        }
    }

    SilicaListView {
        id: contactsView

        anchors {
            top: appBar.bottom
            horizontalCenter: parent.horizontalCenter
            bottom: parent.bottom
        }

        width: root.isTablet ? root.landscapeWidth : root.width
        model: contactsModel
        clip: true
        quickScroll: true

        delegate: ContactBrowserItem {
            id: contactListDelegate

            function getPerson() { return model.person } // access via on-demand call to prevent unnecessary initialization

            anchors {
                left: parent.left
                right: parent.right
            }

            contactId: model.contactId
            searchString: root._searchPattern
            selectionModel: contactSelectionModel

            matchText: typeof model.filterMatchData === "string" ? model.filterMatchData : model.filterMatchData.join(", ")

            onPressed: {
                Qt.inputMethod.hide()
                searchField.focus = false
            }
            onClicked: {
                pageStack.animatorPush("Sailfish.Contacts.ContactCardPage",
                                       {"contact": model.person})
            }
            onContactPressAndHold: {
                root.contactPopupMenu.person = getPerson()
                root.contactPopupMenu.parent = contactListDelegate
                root.contactPopupMenu.multiselect = false

                if (root.isContactsPopupMenu) {
                    constituentFetcher.target = root.contactPopupMenu.person
                    root.contactPopupMenu.person.fetchConstituents()
                }

                root.contactPopupMenu.open()
            }

            highlighted: down || (root.contactPopupMenu && root.contactPopupMenu.parent === contactListDelegate && root.contactPopupMenu.opened)
        }
        VerticalScrollDecorator {}
    }
}
