/*
 * SPDX-FileCopyrightText: 2013–2020 Jolla Ltd.
 * SPDX-FileCopyrightText: 2020-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.6
import Sailfish.Silica 1.0
import org.nemomobile.contacts 1.0
import "recipientfield"
import Sailfish.Contacts 1.0 as Contacts

Item {
    id: root
    property int actionType
    property alias placeholderText: namesList.placeholderText
    //: Summary of all selected recipients, e.g. "Bob, Jane, 75553243"
    //% "Recipients"
    property string summaryPlaceholderText: qsTrId("components_contacts-ph-recipients")
    property alias summary: namesList.summary
    readonly property alias hasFocus: namesList.editing
    property alias fullSummary: namesList.fullSummary
    property QtObject contactSearchModel
    property alias onlineSearchModel: namesList.onlineSearchModel
    property alias onlineSearchDisplayName: namesList.onlineSearchDisplayName
    property bool empty: namesList.summary == ""
    // Supported properties is a combination of: PeopleModel.EmailAddressRequired, AccountUriRequired, PhoneNumberRequired
    property int requiredProperty: PeopleModel.EmailAddressRequired
    property alias multipleAllowed: namesList.multipleAllowed
    property alias inputMethodHints: namesList.inputMethodHints
    property alias recentContactsCategoryMask: namesList.recentContactsCategoryMask
    property int maximumLineCount: 10

    // A model with the following roles:
    // "property" - an object containing the value of the property that the user chose:
    //              a phone number { 'number' }, an email address { 'address' }, or IM account { 'uri', 'path' }
    // "propertyType" - the type of property that the user chose. Either "phoneNumber", "emailAddress" or "accountUri"
    // "formattedNameText" - the name of the contact
    // "person" - the Person object if the user chose from the known contacts
    property QtObject selectedContacts: namesList.recipientsModel

    property QtObject addressesModel: addressesModelId
    property alias showLabel: namesList.showLabel

    property PageStack pickerPageStack: pageStack
    property var pickerPageProperties: ({})

    signal selectionChanged()
    signal lastFieldExited()
    signal pickerRequest()

    function forceActiveFocus() {
        namesList.forceActiveFocus()
    }

    function recipientsToString() {
        return namesList.recipientsToString()
    }

    function setEmailRecipients(addresses) {
        namesList.setEmailRecipients(addresses)
    }

    function _addressList(contact) {
        return ContactsUtil.selectableProperties(contact, requiredProperty, Person)
    }

    function updateSummary() {
        namesList.updateSummary()
    }

    function saveNewContacts() {
        namesList.saveNewContacts()
    }

    onMultipleAllowedChanged: {
        if (!multipleAllowed && namesList.recipientsModel.count > 1) {
            for (var i = namesList.recipientsModel.count - 1; i > 0; i--)
                namesList.recipientsModel.removeRecipient(i)
        }
    }

    implicitHeight: hasFocus ? namesList.height : summaryLabel.height
    height: implicitHeight
    width: parent ? parent.width : Screen.width

    Binding {
        target: contactSearchModel
        property: "requiredProperty"
        value: root.requiredProperty
    }

    ContactPropertyModel {
        id: addressesModelId
        requiredProperty: root.requiredProperty
    }

    AutoCompleteFieldList {
        id: namesList
        requiredProperty: root.requiredProperty
        opacity: editing ? 1.0 : 0.0
        Behavior on opacity { FadeAnimation {} }
        visible: opacity > 0.0

        //: A single recipient
        //% "Recipient"
        placeholderText: qsTrId("components_contacts-ph-recipient")

        onSelectionChanged: root.selectionChanged()
        onLastFieldExited: root.lastFieldExited()
    }

    TextArea {
        id: summaryLabel

        anchors {
            left: parent.left
            leftMargin: Theme.horizontalPageMargin
            right: parent.right
            rightMargin: Theme.horizontalPageMargin
            top: parent.top
            topMargin: Theme.paddingSmall
        }

        height: Math.min(implicitHeight,
                         (root.maximumLineCount + 1) * fm.height +
                         (_labelItem ? _labelItem.height : 0))

        opacity: !root.hasFocus ? 1.0 : 0.0
        visible: opacity > 0.0

        textLeftPadding: 0
        textTopPadding: 0
        textRightPadding: 0
        textMargin: 0
        placeholderText: root.placeholderText
        readOnly: true
        color: summary !== "" ? Theme.primaryColor : Theme.secondaryColor

        text: {
            if (fullSummary != "")
                return fullSummary
            if (summary != "")
                return summary
            return placeholderText
        }

        labelComponent: Item {
            height: labelItem.implicitHeight

            Label {
                id: labelItem

                text: summaryPlaceholderText
                color: Theme.secondaryColor
                visible: summary !== ""
                elide: Text.ElideRight
                font.pixelSize: Theme.fontSizeSmall
            }

            Rectangle {
                width: summaryLabel.width
                height: Theme._lineWidth
                color: Theme.primaryColor
            }
        }

        VerticalAutoScroll.bottomMargin: Theme.paddingMedium

        Behavior on opacity { FadeAnimation {} }

        onClicked: {
            if (!multipleAllowed && summaryLabel.text.length) {
                addressesModel.contact = null
                namesList.recipientsModel.removeRecipient(0, true)
            } else {
                namesList.forceActiveFocus()
            }
        }

        TextMetrics {
             id: fm

             font: summaryLabel.font
             text: summaryLabel.text[0] || " "
         }
    }
}
