/*
 * SPDX-FileCopyrightText: 2012-2020 Jolla Ltd
 * SPDX-FileCopyrightText: 2023-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.6
import Sailfish.Silica 1.0
import Sailfish.Silica.private 1.0
import Sailfish.Contacts 1.0
import org.nemomobile.contacts 1.0

Item {
    id: favoriteBar

    readonly property int _columnsCountForTablets: 6
    readonly property int _columnsCountForPhonesAndSplitView: 4
    readonly property int _maxAvatarSizeForSplitView: Theme.dp(130)
    readonly property bool isLargeScreen: LayoutClassProvider.currentDisplayClass & DisplayLayoutClass.Tablet

    property var favoritesModel
    property ContactSelectionModel selectionModel
    property var disabledModel

    property int columns: {
        // NOTE: This is hack for determine is ContactBrowser in SplitView
        var avatarWidth = parent.width / favoriteBar._columnsCountForPhonesAndSplitView
        var isInSplitView = avatarWidth > favoriteBar._maxAvatarSizeForSplitView
        return favoriteBar.isLargeScreen && isInSplitView
                ? favoriteBar._columnsCountForTablets
                : favoriteBar._columnsCountForPhonesAndSplitView
    }

    property bool menuOpen
    property Item menuItem

    readonly property bool _transitionsEnabled: allowAnimations.running
                                                && !pageStack.currentPage.orientationTransitionRunning

    signal contactClicked(var delegateItem, var contact)
    signal contactPressed()
    signal contactPressAndHold(var delegateItem, var contact)

    width: parent.width
    height: grid.height

    Timer {
        id: allowAnimations
        interval: 400   // roughly after animations complete
    }

    // Only run the add/move animations when the model changes. Otherwise they are triggered
    // when populating the model, or when the grid's overall dimensions change, or a context menu
    // opens within the grid, etc. Using a timer for this is hacky but it's hard to calculate
    // precisely when all add/move transitions have ended e.g. if a transition gets canceled.
    Connections {
        target: favoritesModel
        onCountChanged: {
            if (favoritesModel.populated) {
                allowAnimations.restart()
            }
        }
    }

    Grid {
        id: grid

        anchors {
            left: parent.left
            right: parent.right
        }

        columns: favoriteBar.columns

        add: Transition {
            id: add
            enabled: favoriteBar._transitionsEnabled

            SequentialAnimation {
                PropertyAction {
                    target: add.ViewTransition.item
                    property: "opacity"
                    value: 0
                }
                PauseAnimation {
                    duration: 225
                }
                NumberAnimation {
                    properties: "opacity"
                    duration: 75
                    from: 0
                    to: 1
                }
            }
        }
        move: Transition {
            id: move
            enabled: favoriteBar._transitionsEnabled

            SequentialAnimation {
                id: favoriteItemAnimation

                readonly property bool changingY: {
                    if (!move.ViewTransition.item) {
                        return false
                    }
                    // Animate if the item is moving to first/last column. targetIndexes is only
                    // set when triggered by insertions but not removals, so for insertions check
                    // for a move to the first column, and for removals check for a move to the last.
                    var col = Math.floor(move.ViewTransition.index % grid.columns)
                    return move.ViewTransition.targetIndexes.length > 0 ? col === 0 : col === grid.columns-1
                }

                NumberAnimation {
                    properties: "opacity"
                    duration: 75
                    from: 1
                    to: favoriteItemAnimation.changingY ? 0 : 1
                }
                PauseAnimation {
                    duration: favoriteItemAnimation.changingY ? 150 : 0
                }
                NumberAnimation {
                    properties: "x,y"
                    easing.type: Easing.InOutQuad
                    duration: favoriteItemAnimation.changingY ? 0 : 150
                }
                NumberAnimation {
                    properties: "opacity"
                    duration: 75
                    from: favoriteItemAnimation.changingY ? 0 : 1
                    to: 1
                }
            }
        }

        Repeater {
            id: repeater

            model: favoritesModel

            FavoriteContactItem {
                id: contactItem

                width: grid.width / favoriteBar.columns
                selectionModel: favoriteBar.selectionModel
                disabledModel: favoriteBar.disabledModel
                acceptedButtons: Qt.LeftButton | Qt.RightButton
                openMenuOnPressAndHold: false
                highlighted: down || menuOpen || selectionModelIndex >= 0

                onPressed: contactPressed()

                onPressAndHold: contactPressAndHold(contactItem, model.person)

                onClicked: {
                    if (mouse.button === Qt.RightButton) {
                        contactPressAndHold(contactItem, model.person)
                        return
                    }

                    contactClicked(contactItem, model.person)
                }

                onMenuOpenChanged: {
                    if (menuOpen) {
                        favoriteBar.menuItem = _menuItem
                    } else if (favoriteBar.menuItem === _menuItem) {
                        favoriteBar.menuItem = null
                    }
                    favoriteBar.menuOpen = menuOpen
                }
            }
        }
    }
}
