/*
 * SPDX-FileCopyrightText: 2012-2019 Jolla Ltd
 * SPDX-FileCopyrightText: 2020-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.4
import Sailfish.Silica 1.0
import Sailfish.Contacts 1.0 as Contacts
import org.nemomobile.contacts 1.0

GridItem {
    id: favoriteItem

    readonly property int contactId: model.contactId
    property bool canDeleteContact: true
    property var selectionModel
    readonly property int selectionModelIndex: {
        if (selectionModel !== null && selectionModel.count > 0) {
            var ids = selectionModel.findContactId(contactId)
            if (ids.length > 0) return ids[0]
        }
        return -1 // count to retrigger on change.
    }
    property var disabledModel

    property var propertyPicker
    property var disabledItems: (disabledModel && disabledModel.count)
                                ? disabledModel.getDisabledItems(contactId) : null

    property bool _hasAvatar: model.avatarUrl != ""

    property bool _pendingDeletion: Contacts.ContactModelCache._deletingContactId === contactId

    // NOTE:  -2 * Theme.paddingMedium - margins of right and left side
    readonly property int itemSize: favoriteItem.width - 2 * Theme.paddingMedium

    function personObject() {
        return model.person
    }

    function deleteContact() {
        var item = remorseDelete(function () {
            Contacts.ContactModelCache.deleteContact(person)
        })
        if (openRemorseBelow) {
            item.rightMargin = Theme.paddingMedium
        }
    }

    function getAbbreviation() {
        var abbreviation = ""
        if (model.person) {
            if (model.person.primaryName.length > 0) {
                abbreviation += model.person.primaryName.charAt(0).toUpperCase()
            }
            if (model.person.secondaryName.length > 0) {
                abbreviation += model.person.secondaryName.charAt(0).toUpperCase()
            }
        }
        return abbreviation
    }

    opacity: _pendingDeletion ? 0.0 : (enabled ? 1.0 : Theme.opacityLow)
    contentHeight: width + primaryNameLabel.height + secondaryNameLabel.height + Theme.paddingMedium
    highlighted: down || menuOpen || selectionModelIndex >= 0
    highlightedColor: "transparent"
    _backgroundColor: "transparent"
    contentItem.clip: !_hasAvatar
    enabled: disabledItems ? model.phoneNumbers.length !== disabledItems.length : true

    Item {
        id: favoriteIcon

        anchors {
            top: parent.top
            horizontalCenter: parent.horizontalCenter
        }

        height: favoriteItem.itemSize
        width: favoriteItem.itemSize

        Rectangle {
            anchors.fill: parent
            color: Theme.rgba(Theme.primaryColor, Theme.opacityFaint)
            radius: width / 2
            visible: !_hasAvatar

            Label {
                id: label

                anchors.centerIn: parent
                text: getAbbreviation()
                font.pixelSize: Theme.fontSizeExtraLarge
                color: Theme.primaryColor
                truncationMode: TruncationMode.Fade
                maximumLineCount: 1
                objectName: "label"
            }
        }

        RoundedImage {
            anchors.fill: parent
            image.source: model.avatarUrl
        }
    }

    Label {
        id: primaryNameLabel

        anchors {
            top: favoriteIcon.bottom
            topMargin: text.length > 0 ? Theme.paddingSmall : 0
        }
        text: Contacts.ContactModelCache.getPersonFirstText(model.person)
        width: parent.width
        height: text.length > 0 ? implicitHeight : 0
        font {
            family: Theme.fontFamilyHeading
            pixelSize: Theme.fontSizeTiny
        }
        horizontalAlignment: Text.AlignHCenter
        elide: Text.ElideRight
        maximumLineCount: 1
    }

    Label {
        id: secondaryNameLabel

        anchors {
            top: primaryNameLabel.bottom
            topMargin: Theme.paddingSmall
        }

        text: Contacts.ContactModelCache.getPersonSecondText(model.person)
        width: parent.width
        height: text.length > 0 ? implicitHeight : 0
        font {
            family: Theme.fontFamilyHeading
            pixelSize: Theme.fontSizeTiny
        }
        horizontalAlignment: Text.AlignHCenter
        elide: Text.ElideRight
        maximumLineCount: 1
    }
}
