/*
 * SPDX-FileCopyrightText: Copyright 2022-2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 *
 * Copyright (c) 2019 Jolla Ltd.
 *
 * License: Proprietary
 */

import QtQuick 2.0
import Sailfish.Silica 1.0
import org.nemomobile.contacts 1.0
import Aurora.Controls 1.0
import Aurora.Applications.Internal 1.0 as AppInternal
import "."

QtObject {
    id: root

    property var silicaListView
    readonly property bool pickerPageActive: false // deprecated.

    property var contactDelegateItem
    property alias contact: propertyModel.contact
    property alias disabledItems: propertyModel.disabledItems
    property alias requiredProperty: propertyModel.requiredProperty
    property var propertySelectedCallback
    property bool closeOnSelection: true
    readonly property bool phoneNumberRequired: root.requiredProperty & PeopleModel.PhoneNumberRequired

    property PopupMenu _popupMenu: null

    property var _propertyModel: ContactPropertyModel {
        id: propertyModel
    }

    property Component _popupMenuComponent: Component {
        PopupMenu {
            id: self

            property bool preopened: false

            parent: root.contactDelegateItem

            Repeater {
                model: _propertyModel

                PopupMenuItem {
                    text: {
                        if (root.phoneNumberRequired) {
                            try {
                                return model.displayLabel
                                            .split("|")[0].trim()
                                            .split(ContactsUtil.labelDisplaySeparator)[0]
                            } catch (e) {}
                        }

                        return model.displayLabel
                    }

                    hint: root.phoneNumberRequired ? model.property.number : ""

                    enabled: !model.disabled

                    onClicked: root._propertySelected(propertyModel.get(model.index), self)
                }
            }

            onOpenedChanged: {
                if (!self.opened) {
                    self.preopened = false
                }
            }

            onClosed: {
                if (root.closeOnSelection) {
                    self.destroy()
                }
            }

            Component.onCompleted: {
                self.preferredWidth = AppInternal.UiUtils.getPopupMenuPreferredWidth({
                    popupMenu: self,
                    minimumWidth: AppInternal.UiUtils.isLargeScreen
                                  ? self.preferredWidth
                                  : silicaListView.width,
                    maximumWidth: silicaListView.width
                })
            }
        }
    }

    readonly property Item __logic: Item {
        readonly property bool popupMenuOpened: root._popupMenu && root._popupMenu.preopened

        /*
          Using Binding coz we dont know what value of highlighted was before.
          Binding will restore it for us so we dont realy care.
         */
        Binding {
            target: root.contactDelegateItem
            when: __logic.popupMenuOpened
            property: 'highlighted'
            value: __logic.popupMenuOpened
        }
    }

    function openMenu() {
        if (!root.contactDelegateItem) {
            console.warn("Error: cannot load property picker, delegate item not set!")
            return
        }

        if (!root.contact) {
            console.warn("Error: cannot load property picker, contact not set!")
            return
        }

        if (root.requiredProperty === PeopleModel.NoPropertyRequired) {
            console.warn("Error: cannot load property picker, requiredProperty not set!")
            return
        }

        root.disabledItems = contactDelegateItem.disabledItems

        if (propertyModel.count === 0) {
            root._propertySelected({}, root._popupMenu)
        } else if (propertyModel.count === 1) {
            // There's only one selectable property, so select it immediately instead of
            // showing the picker.
            root._propertySelected(propertyModel.get(0), root._popupMenu)
        } else {
            // Open the property picker in the currently active context menu if found, or
            // otherwise open a new context menu.
            if (!root._popupMenu) {
                root._popupMenu = _popupMenuComponent.createObject(root.contactDelegateItem)
            }

            /* Dont wait for Aurora::PopupMenu::progress before
               Aurora::PopupMenu::opened become true.
               Avoid contact delegate blinking. */
            root._popupMenu.preopened = true
            root._popupMenu.open()
        }
    }

    function openPickerPageMenu(menu, menuProperties) {
        console.warn("this function not used anymore and will be deleted")
    }

    function closeMenu() {
        if (!root._popupMenu || !root._popupMenu.opened) {
            return
        }

        root._popupMenu.close()
    }

    function _propertySelected(propertyData, contextMenu) {
        if (root.contactDelegateItem.selectionModel &&
                root.contactDelegateItem.selectionModelIndex < 0) {
            root.contactDelegateItem.selectionModel.addContact(
                        contactDelegateItem.contactId, propertyData.property, propertyData.propertyType)
        }

        root.propertySelectedCallback(contact, propertyData, contextMenu, root)

        if (root.closeOnSelection) {
            root.closeMenu()
        }
    }
}
