/*
 * SPDX-FileCopyrightText: 2016-2020 Jolla Ltd
 * SPDX-FileCopyrightText: 2023-2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.5
import Aurora.Controls 1.0
import Sailfish.Silica 1.0
import Sailfish.Silica.private 1.0
import Sailfish.Contacts 1.0
import org.nemomobile.contacts 1.0

Page {
    id: root
    allowedOrientations: Orientation.All

    property ContactSelectionModel selectedContacts: contactBrowser.selectedContacts
    property alias requiredProperty: contactBrowser.requiredContactProperty
    property alias recentContactsCategoryMask: contactBrowser.recentContactsCategoryMask
    property alias searchActive: contactBrowser.searchActive
    property int blockedContactsSelected: 0
    property bool forceToRecacheRemovableContacts
    readonly property bool enableButtons: selectedContacts.count > 0
    readonly property var _peopleModel: ContactModelCache.unfilteredModel()

    readonly property bool isTablet: LayoutClassProvider.currentDisplayClass & DisplayLayoutClass.Tablet
    readonly property bool isPortrait: orientation & Orientation.PortraitMask
    readonly property real landscapeWidth: root.isPortrait ? root.width : Screen.width

    signal contactClicked(var contact)
    signal shareClicked(var content)
    signal deleteClicked(var contactIds)
    signal blockClicked(var contactIds)

    function _deleteSelection() {
        root.deleteClicked(selectedContacts.allContactIds())
    }

    function _shareSelection() {
        // share all of the selected contacts
        var vcardName = "" + root.selectedContacts.count + "-contacts.vcf"
        var vcardData = ""
        for (var i = 0; i < root.selectedContacts.count; ++i) {
            vcardData = vcardData + contactBrowser.allContactsModel.personById(root.selectedContacts.get(i)).vCard()
        }
        vcardData = vcardData + "\r\n"
        var content = {
            "data": vcardData,
            "name": vcardName,
            "type": "text/vcard"
        }
        root.shareClicked(content)
    }

    function _blockSelection() {
        root.blockClicked(selectedContacts.allContactIds())
    }

    function _doSelectionOperation(selectAll) {
        contactBrowser.selectedContacts.removeAllContacts()
        if (selectAll) {
            var currentIndex = 0
            var lastIndex = contactBrowser.allContactsModel.count - 1
            while (currentIndex <= lastIndex) {
                contactBrowser.selectedContacts.addContact(
                        contactBrowser.allContactsModel.get(
                                currentIndex,
                                PeopleModel.ContactIdRole),
                        null, null,
                        currentIndex != lastIndex
                                ? ContactSelectionModel.BatchMode
                                : ContactSelectionModel.SingleContactMode)
                currentIndex++
            }
        }
    }

    function _handlingOfCacheReady(readonlyContactsCount) {
        readonlyContactsSelected = readonlyContactsCount
        deleteButtonBusyIndicator.running = false
    }

    AppBar {
        id: appBar

        headerText: root.selectedContacts.count > 0
        //: Indicates number of selected contacts
        //% "%n selected"
        ? qsTrId("components_pickers-la-count_selected", root.selectedContacts.count)
          //: Hint that the user should select contacts
          //% "Select contacts"
        : qsTrId("components_pickers-la-select_contacts")
        visible: !main.addCallMode

        AppBarSpacer {}
        AppBarButton {
            icon.source: "image://theme/icon-splus-multiselect"
            onClicked: {
                _doSelectionOperation(true)
            }
        }
        AppBarButton {
            icon.source: "image://theme/icon-splus-multiclean"
            onClicked: {
                _doSelectionOperation(false)
            }
        }
    }

    AppBar {
        id: bottomAppBar

        _bottomPosition: true

        AppBarSpacer {}
        AppBarButton {
            icon.source: "image://theme/icon-splus-blocked"

            enabled: root.enableButtons && root.blockedContactsSelected === 0
            onClicked: root._blockSelection()
        }
        AppBarSpacer {}
        AppBarButton {
            icon.source: "image://theme/icon-m-share"

            enabled: root.enableButtons
            onClicked: root._shareSelection()
        }
        AppBarSpacer {}
        AppBarButton {
            icon.source: "image://theme/icon-splus-delete"

            enabled: root.enableButtons

            onClicked: {
                root._deleteSelection()
            }
        }
        AppBarSpacer {}
    }

    ContactBrowser {
        id: contactBrowser

        anchors {
            top: appBar.bottom
            bottom: bottomAppBar.top
            horizontalCenter: parent.horizontalCenter
        }
        clip: true
        height: undefined // reset height binding to allow use anchors insteaed
        width: root.isTablet ? root.landscapeWidth : root.width
        canSelect: true
        showContacts: true
        showFavoriteContacts: false
        showRecentContacts: false

        onContactClicked: {
            if (contactBrowser.allContactsModel.isBlockedPerson(contact)) {
                root.blockedContactsSelected++
            }
            root.contactClicked(contact)
        }

        onSelectionRemoved: {
            if (contactBrowser.allContactsModel.isBlockedPerson(contact)) {
                root.blockedContactsSelected--
            }
        }
    }
}
