/*
 * SPDX-FileCopyrightText: Copyright 2020-2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 *
 * Copyright (c) 2013 - 2019 Jolla Ltd.
 *
 * License: Proprietary
 */

pragma Singleton
import QtQuick 2.6
import Sailfish.Silica 1.0
import org.nemomobile.contacts 1.0

QtObject {
    id: root

    property var _unfilteredModel
    property int _deletingContactId: -1
    property var _constituentFetchers: []

    function unfilteredModel() {
        if (_unfilteredModel == null) {
            _unfilteredModel = _unfilteredModelComponent.createObject(root)
        }
        return _unfilteredModel
    }

    property Component _unfilteredModelComponent: Component {
        PeopleModel {
            filterType: PeopleModel.FilterAll

            // Can't make the object read-only, so discourage changes that result in model filtering.
            onFilterTypeChanged: console.log("ContactModelCache.unfilteredModel() should not be filtered! Create another PeopleModel instead.")
            onFilterPatternChanged: console.log("ContactModelCache.unfilteredModel() should not be filtered! Create another PeopleModel instead.")
            onRequiredPropertyChanged: console.log("ContactModelCache.unfilteredModel() should not have requiredProperty set! Create another PeopleModel instead.")
            onSearchablePropertyChanged: console.log("ContactModelCache.unfilteredModel() should not have searchableProperty set! Create another PeopleModel instead.")

            onContactBlocked: console.log("ContactModelCache.unfilteredModel() contact has been blocked")
        }
    }

    function deleteContact(contact, args) {
        if (!args) {
            args = {}
        }
        var success = true

        if (contact.addressBook.isAggregate) {
            var fetcher = _constituentFetchAndDeleteComponent.createObject(
                        root, { "target": contact })
            _constituentFetchers.push(fetcher)
            contact.fetchConstituents()

            if (args.successCallback) {
                args.successCallback()
            }
        } else if (!contact.addressBook.readOnly) {
            unfilteredModel().removePerson(contact)

            if (args.successCallback) {
                args.successCallback()
            }
        } else {
            const message = "Ignoring request to delete read-only, non-aggregate contact"

            console.warn(message, contact.id, contact.displayLabel)
            success = false

            if (args.errorCallback) {
                args.errorCallback(message)
            }
        }

        if (success) {
            Notices.show(qsTrId("contacts-me-removed_contact"), Notice.Short, Notice.Bottom)
        }
    }

    function blockContact(value) {
        if (value && value.id) {
            return unfilteredModel().blockPerson(value)
        } else if (value) {
            return unfilteredModel().blockPhoneNumber(value)
        }
        return false
    }

    function unblockContact(value) {
        if (value && value.id) {
            return unfilteredModel().unBlockPerson(value)
        } else if (value) {
            return unfilteredModel().unBlockPhoneNumber(value)
        }
        return false
    }

    function isBlockedContact(value) {
        if (value && value.id) {
            return unfilteredModel().isBlockedPerson(value)
        } else if (value) {
            return unfilteredModel().isBlockedPhoneNumber(value)
        }
        return false
    }

    function getBlockedContacts() {
        return unfilteredModel().blockedContacts();
    }

    function _deleteConstituents(constituents) {
        var people = []
        for (var i = 0; i < constituents.length; ++i) {
            var person = unfilteredModel().personById(constituents[i])
            if (!person.addressBook.isReadOnly) {
                people.push(person)
            }
        }
        if (people.length > 0) {
            unfilteredModel().removePeople(people)
        } else {
            console.warn("All contacts belong to read-only address books, so will not be deleted")
        }
    }

    function getPersonFirstText(person) {
        if (!person) {
            return ""
        }

        const firstNameFirst = getPersonFirstNameFirst()

        if (firstNameFirst && person.firstName && person.middleName) {
            return [person.primaryName, person.middleName].join(" ")
        }

        if (!firstNameFirst && person.secondaryName && person.firstName) {
            return [person.primaryName, person.secondaryName].join(" ")
        }

        if (firstNameFirst) {
            return person.primaryName || person.middleName
        }

        return person.secondaryName || person.primaryName
    }

    function getPersonSecondText(person) {
        if (!person) {
            return ""
        }

        const firstNameFirst = getPersonFirstNameFirst()

        if (firstNameFirst && person.secondaryName) {
            return person.secondaryName
        }

        if (!firstNameFirst && person.secondaryName) {
            return person.middleName
        }

        return ""
    }

    function getPersonFirstNameFirst() {
        return root.unfilteredModel().displayLabelOrder === PeopleModel.FirstNameFirst
    }

    property var _constituentFetchAndDeleteComponent: Component {
        Connections {
            id: constituentFetchAndDelete

            onConstituentsChanged: {
                _deleteConstituents(target.constituents)
                for (var i = 0; i < root._constituentFetchers.length; ++i) {
                    if (root._constituentFetchers[i] === constituentFetchAndDelete) {
                        root._constituentFetchers.splice(i, 1)
                        break
                    }
                }
                destroy()
            }
        }
    }
}
