/*
 * SPDX-FileCopyrightText: Copyright 2023 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary

* Copyright (c) 2020 - 2023 Open Mobile Platform LLC.
* Copyright (c) 2013 - 2020 Jolla Pty Ltd.
*
* License: Proprietary
*/

import QtQuick 2.6
import Sailfish.Silica 1.0
import org.nemomobile.contacts 1.0

ListItem {
    id: root

    property alias firstText: row.firstText
    property alias secondText: row.secondText
    property string matchText
    property alias unnamed: row.unnamed
    property int presenceState
    property alias iconSource: icon.source
    property alias isBlocked: row.showBlockedIcon

    property string searchString

    property real leftMargin: Theme.paddingMedium
    property real leftMarginOffset

    // Same as: SearchField.textLeftMargin
    property real searchLeftMargin: contactItemLeftMargin > 0 ? contactItemLeftMargin : Theme.itemSizeSmall + Theme.paddingMedium

    property bool _matchTextVisible: searchString.length > 0 && matchText.length > 0 && firstText != matchText

    property var _searchWords: []
    property string _trimmedStr

    contentHeight: _matchTextVisible ? Theme.itemSizeMedium : Theme.itemSizeSmall

    highlightedColor: Theme.rgba(Theme.primaryColor, 0.2)
    _backgroundRadius: Theme.dp(16)

    function _regExpFor(term) {
        var symbols = RegExp(/([.?*+^$[\]\\(){}|-])/g)

        var exp = RegExp(_trimmedStr.replace(symbols, "\\$1"), 'i')
        if (term.search(exp) !== -1) {
            return exp
        }
        for (var i = 0; i < _searchWords.length; i++) {
            exp = RegExp(_searchWords[i].replace(symbols, "\\$1"), 'i')
            if (term.search(exp) !== -1) {
                return exp
            }
        }
        return new RegExp("")
    }

    onSearchStringChanged: {
        _trimmedStr = searchString.replace(/\s+/g, ' ')
        _searchWords = _trimmedStr.split(' ')
    }

    ContactNameRow {
        id: row

        anchors {
            left: parent.left
            right: icon.left
            leftMargin: Theme.paddingMedium
            rightMargin: icon.width ? spacing : 0
            verticalCenter: parent.verticalCenter
            verticalCenterOffset: _matchTextVisible ? -matchDataText.height/2 : 0
        }
    }

    Label {
        id: matchDataText

        anchors {
            left: row.left
            right: row.right
            top: row.bottom
        }

        text: matchText
        color: highlighted ? Theme.secondaryHighlightColor: Theme.secondaryColor
        truncationMode: TruncationMode.Fade
        textFormat: Text.AutoText
        visible: _matchTextVisible
    }

    Image {
        id: icon

        anchors {
            right: parent.right
            rightMargin: Theme.horizontalPageMargin
            verticalCenter: parent.verticalCenter
        }

        // use app icons, scaled to half their size
        width: visible ? Theme.itemSizeMedium / 2 : 0
        height: Theme.itemSizeMedium / 2
        source: iconSource
        visible: iconSource != ''
    }
    ContactPresenceIndicator {
        id: presence
        visible: !offline
        anchors {
            top: row.top
            topMargin: row.firstNameLabel.baselineOffset + Theme.paddingMedium
            left: row.left
            right: undefined
        }
        animationEnabled: false
        presenceState: root.presenceState
    }

    states: State {
        when: searchString !== ''
        PropertyChanges {
            target: root
            leftMargin: root.searchLeftMargin
        }
        PropertyChanges {
            target: row.firstNameLabel
            text: Theme.highlightText(firstText, _regExpFor(firstText), Theme.highlightColor)
            textFormat: Text.StyledText
        }
        PropertyChanges {
            target: row.lastNameLabel
            text: Theme.highlightText(secondText, _regExpFor(secondText), Theme.highlightColor)
            textFormat: Text.StyledText
        }
        PropertyChanges {
            target: matchDataText
            text: Theme.highlightText(matchText, _regExpFor(matchText), Theme.highlightColor)
            textFormat: Text.StyledText
        }
    }
}
