/*
 * SPDX-FileCopyrightText: 2013-2018 Jolla Ltd
 * SPDX-FileCopyrightText: 2023-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.0
import Sailfish.Silica 1.0
import Sailfish.Contacts 1.0
import Sailfish.Silica.private 1.0
import Sailfish.Accounts 1.0
import com.jolla.settings.accounts 1.0
import MeeGo.QOfono 0.2
import Aurora.Controls 1.0

Page {
    id: root

    property string importSourceUrl
    property string importSourceModemPath
    property bool accountImport
    property AccountCreationManager accountCreator
    property Page rootPage

    signal contactOpenRequested(var contactId)
    signal goBack()

    //=== internal/private members follow

    property string _fileName
    property bool _fileImport: _fileName != ''
    property int _readCount: 0
    property int _savedCount: 0
    property int _ignoredCount: 0
    property int _updatedCount: 0
    property bool _error
    property bool _noContacts
    property var _importedContactId
    property bool _simImportStarted
    property bool _endSync

    readonly property bool isTablet: LayoutClassProvider.currentDisplayClass & DisplayLayoutClass.Tablet
    readonly property bool isPortrait: orientation & Orientation.PortraitMask

    onStatusChanged: {
        if (status === PageStatus.Deactivating) {
            versitImport.cancel()
        }
    }

    Component.onCompleted: {
        if (importSourceUrl != "") {
            busyIndicator.running = true
            versitImport.importContacts(_addFileScheme(importSourceUrl))
            var index = importSourceUrl.lastIndexOf("/")
            _fileName = decodeURIComponent(index >= 0 ? importSourceUrl.substring(index+1) : importSourceUrl)
        } else if (importSourceModemPath != "") {
            busyIndicator.running = true
            // This beginImport() call won't actually do anything if
            // OfonoPhonebook isn't ready yet. In that case we actually
            // start the import in OfonoPhonebook.onValidChanged handler.
            // The _simImportStarted property ensures that we don't start
            // the import more than once.
            ofonoPhonebook.beginImport()
        } else if (accountImport) {
            busyIndicator.running = true
        }
    }

    function _startsWith(str, word) {
        return str.lastIndexOf(word, 0) === 0
    }

    function _addFileScheme(url) {
        const fileScheme = 'file:///'

        if (!_startsWith(url, fileScheme)) {
            url = fileScheme.concat('', url)
        }

        return url
    }

    function _statusText() {
        if (_savedCount > 0) {
            if (_fileImport) {
                //% "Imported %n new contact(s) found in %1."
                return qsTrId("contacts_settings-la-imported_n_contacts_file", _savedCount).arg(root._fileName)
            } else if (accountImport) {
                //% "Imported %n new contact(s) found on Exchange account."
                return qsTrId("contacts_settings-la-imported_n_contacts_exchange", _savedCount)
            } else {
                //% "Imported %n new contact(s) found on SIM card."
                return qsTrId("contacts_settings-la-imported_n_contacts_sim", _savedCount)
            }
        } else if (_readCount > 0 && _savedCount == 0) {
            //% "No new contacts to import."
            return qsTrId("contacts_settings-la-no_new_contacts")
        } else {
            //% "Contacts imported."
            return qsTrId("components_contacts-la-contacts_imported")
        }
    }

    function _extraStatusText() {
        var tmp = []
        if (root._updatedCount > 0) {
            //% "updated %n"
            tmp.push(qsTrId("components_contacts-la-updated", root._updatedCount))
        }
        if (root._ignoredCount > 0) {
            //% "skipped %n"
            tmp.push(qsTrId("components_contacts-la-skipped", root._ignoredCount))
        }
        var result = tmp.join(", ")
        //% "contacts."
        result = result.concat(" ", qsTrId("components_contacts-contacts"))

        return String(result).charAt(0).toUpperCase() + String(result).substring(1)
    }

    function _noContactsText() {
        if (_fileImport) {
            //% "There are no contacts in %1."
            return qsTrId("components_contacts-la-no_contacts_in_file").arg(root._fileName)
        } if (accountImport) {
            //% "There are no contacts on your Exchange account."
            return qsTrId("components_contacts-la-no_contacts_on_exchange")
        } else {
            //% "There are no contacts on your Sim card."
            return qsTrId("components_contacts-la-no_contacts_on_sim")
        }
    }

    function _fillContactsImportResults(results) {
        if (Object.keys(results).length > 0) {
            root._readCount = results["readedContacts"]
            root._savedCount = results["newContacts"]
            root._ignoredCount = results["ignoredContacts"]
            root._updatedCount = results["updatedContacts"]
        }
    }

    Connections {
        target: accountCreator

        onProfileSyncStatusChanged: {
            if (profileId.indexOf("sailfisheas.Contacts") !== -1) {
                if (status !== AccountSyncManager.SyncStarted && !_endSync) {
                    if (!_savedCount) {
                        _noContacts = true
                    }

                    if (status === AccountSyncManager.SyncError) {
                        _error = true
                    }

                    busyIndicator.running = false
                    _endSync = true
                }
            }
        }
    }

    AppBar {
        id: appBar
    }

    Column {
        anchors {
            top: appBar.bottom
            topMargin: Theme.paddingMedium
            horizontalCenter: parent.horizontalCenter
        }

        width: root.isLandscape && root.isTablet ? Screen.width : (root.width - Theme.paddingLarge * 2)
        spacing: Theme.paddingLarge

        Label {
            id: header

            width: parent.width
            color: Theme.primaryColor
            font.pixelSize: Theme.fontSizeExtraLarge
            wrapMode: Text.Wrap
        }

        Label {
            id: status

            width: parent.width
            color: Theme.rgba(Theme.primaryColor, Theme.opacityOverlay)
            wrapMode: Text.Wrap
        }

        Label {
            id: extraImportingStatus

            width: parent.width
            color: Theme.rgba(Theme.primaryColor, Theme.opacityOverlay)
            wrapMode: Text.Wrap
            visible: root._updatedCount > 0 || root._ignoredCount > 0
        }

        BusyIndicator {
            id: busyIndicator

            anchors.horizontalCenter: parent.horizontalCenter
            size: BusyIndicatorSize.Large
        }
    }

    Button {
        id: viewButton

        anchors {
            bottom: parent.bottom
            bottomMargin: Theme.itemSizeMedium
            horizontalCenter: parent.horizontalCenter
        }

        visible: false
        text: root._importedContactId !== undefined
                //% "View contact"
              ? qsTrId("components_contacts-la-view_contact")
                //% "View all contacts"
              : qsTrId("components_contacts-la-view_all_contact")

        Behavior on visible { FadeAnimation {} }

        onClicked: {
            if (root.accountImport) {
                pageStack.pop(rootPage)
            } else {
                root.contactOpenRequested(root._importedContactId)
            }
        }
    }

    Button {
        id: backButton

        anchors {
            bottom: parent.bottom
            bottomMargin: Theme.itemSizeMedium
            horizontalCenter: parent.horizontalCenter
        }

        visible: false

        Behavior on visible { FadeAnimation {} }

        onClicked: {
            if (root.state === "noContacts")
                goBack()
            else
                pageStack.pop(rootPage)
        }
    }

    Button {
        id: tryAgainButton

        anchors {
            bottom: backButton.top
            bottomMargin: Theme.itemSizeMedium
            horizontalCenter: parent.horizontalCenter
        }

        visible: false
        //% "Try again"
        text: qsTrId("components_contacts-la-try_again")

        Behavior on visible { FadeAnimation {} }

        onClicked: {
            busyIndicator.running = true
            ofonoPhonebook.beginImport()
        }
    }

    ContactImporter {
        id: versitImport

        onImportFinished: {
            _fillContactsImportResults(results)

            if (root._savedCount == 1) {
                fetchFirstSavedContact()
            } else {
                _noContacts = !root._readCount && !root._savedCount
                busyIndicator.running = false
            }
        }

        onImportFailed: {
            busyIndicator.running = false
            root._error = true
        }
        onFetchFirstSavedContactFinished: {
            root._importedContactId = firstSavedContactId
            busyIndicator.running = false
        }

        onFetchFirstSavedContactFailed: busyIndicator.running = false

        onFetchStateChanged: {
            if (root.accountImport && fetchState !== 1) {
                _savedCount = fetchedCount
                _noContacts = !fetchedCount
            }
        }
    }

    OfonoPhonebook {
        id: ofonoPhonebook

        modemPath: root.importSourceModemPath

        onImportingChanged: if (importing) _simImportStarted = true
        onValidChanged: if (valid && !_simImportStarted) ofonoPhonebook.beginImport()

        onImportReady: {
            if(vcardData.length !== 0)
                versitImport.importContactsData(vcardData)
            else {
                busyIndicator.running = false
                _noContacts = true
            }
        }

        onImportFailed: {
            busyIndicator.running = false
            root._error = true
        }
    }

    states: [
        State {
            name: "fileImportActive"
            when: busyIndicator.running && _fileImport
            PropertyChanges {
                target: header
                //% "Importing contacts"
                text: qsTrId("components_contacts-he-importing_contacts")
            }
            PropertyChanges {
                target: status
                //: Importing contacts from a specified file
                //% "Importing contacts from %1..."
                text: qsTrId("components_contacts-la-importing_contacts_from_file").arg(_fileName)
            }
        },
        State {
            name: "simImportActive"
            when: busyIndicator.running && !_fileImport && !accountImport
            PropertyChanges {
                target: header
                //% "Importing contacts"
                text: qsTrId("components_contacts-he-importing_contacts")
            }
            PropertyChanges {
                target: status
                //: Importing contacts from SIM
                //% "Importing contacts from SIM card..."
                text: qsTrId("components_contacts-la-importing_contacts_from_sim")
            }
        },
        State {
            name: "accountImportActive"
            when: busyIndicator.running && accountImport
            PropertyChanges {
                target: header
                //% "Importing contacts"
                text: qsTrId("components_contacts-he-importing_contacts")
            }
            PropertyChanges {
                target: status
                //: Importing contacts from Exchange account
                //% "Importing contacts from Exchange account..."
                text: qsTrId("components_contacts-la-importing_contacts_from_exchange")
            }
        },
        State {
            name: "fileImportError"
            when: !busyIndicator.running && _fileImport && _error
            PropertyChanges {
                target: header
                //% "Import error"
                text: qsTrId("components_contacts-he-import_error")
            }
            PropertyChanges {
                target: status
                //: Error while importing contacts from the specified file
                //% "Unable to import contacts from %1."
                text: qsTrId("components_contacts-la-failed_to_import_contacts_file").arg(_fileName)
            }
            PropertyChanges {
                target: backButton
                visible: true
                //% "Cancel"
                text: qsTrId("components_contacts-la-cancel")
            }
        },
        State {
            name: "simImportError"
            when: !busyIndicator.running && !_fileImport && !accountImport && _error
            PropertyChanges {
                target: header
                //% "Import error"
                text: qsTrId("components_contacts-he-import_error")
            }
            PropertyChanges {
                target: status
                //: Error while importing contacts from SIM
                //% "Error while importing contacts from SIM. Try again later."
                text: qsTrId("components_contacts-la-failed_to_import_contacts_sim")
            }
            PropertyChanges { target: tryAgainButton; visible: true }
            PropertyChanges {
                target: backButton
                visible: true
                //% "Cancel"
                text: qsTrId("components_contacts-la-cancel")
            }
        },
        State {
            name: "accountImportError"
            when: !busyIndicator.running && accountImport && _error
            PropertyChanges {
                target: header
                //% "Import error"
                text: qsTrId("components_contacts-he-import_error")
            }
            PropertyChanges {
                target: status
                //: Error while importing contacts from Exchange account
                //% "Unable to import contacts from Exchange account."
                text: qsTrId("components_contacts-la-failed_to_import_contacts_exchange").arg(_fileName)
            }
            PropertyChanges {
                target: backButton
                visible: true
                //% "Cancel"
                text: qsTrId("components_contacts-la-cancel")
            }
        },
        State {
            name: "noContacts"
            when: !busyIndicator.running && _noContacts && !_error
            PropertyChanges {
                target: header
                //% "Import is impossible"
                text: qsTrId("components_contacts-he-import_impossible")
            }
            PropertyChanges {
                target: status
                text: root._noContactsText()
            }
            PropertyChanges {
                target: backButton
                visible: true
                //% "Go back"
                text: qsTrId("components_contacts-la-go_back")
            }
        },
        State {
            name: "importFinished"
            when: !busyIndicator.running && !_error && !_noContacts
            PropertyChanges {
                target: header
                //% "Import finished"
                text: qsTrId("components_contacts-he-import_finished")
            }
            PropertyChanges {
                target: status
                text: root._statusText()
            }
            PropertyChanges {
                target: extraImportingStatus
                text: _extraStatusText()
            }
            PropertyChanges { target: viewButton; visible: true }
        }
    ]
}
