/*
 * SPDX-FileCopyrightText: 2012-2019 Jolla Ltd
 * SPDX-FileCopyrightText: 2023-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
*/

import QtQuick 2.6
import Aurora.Controls 1.0
import Sailfish.Silica 1.0
import Sailfish.Telephony 1.0
import Sailfish.Contacts 1.0 as SailfishContacts
import org.nemomobile.contacts 1.0

Column {
    id: root

    property var contact
    property var simManager

    property bool contactIsBlocked: false
    property bool unsavedContact

    function getNameText() {
        if (contact) {
            if (PeopleModelUtils.isNumber(contact.displayLabel)) {
                return AsYouTypeFormatterUtil.getFormattedPhoneNumber(contact.displayLabel)
            } else {
                return contact.displayLabel
            }
        }
        return ''
    }

    function getDetailText() {
        if (contact && contact.complete) {
            var items = []
            // TODO: find the 'preferred' nickname
            var nicknames = contact.nicknameDetails
            for (var i = 0; i < nicknames.length; ++i) {
                // If the contact nickname is already the display label, don't show it here
                if (nicknames[i].nickname != getNameText()) {
                    items.push(nicknames[i].nickname)
                    // Only use one nickname
                    break
                }
            }
            if (contact.companyName) {
                // Don't show company name if it duplicates the displayed name
                if (contact.companyName != getNameText()) {
                    items.push(contact.companyName)
                }
            }
            if (contact.department) {
                items.push(contact.department)
            }
            if (contact.title) {
                items.push(contact.title)
            }
            if (contact.role) {
                items.push(contact.role)
            }
            return items.join(', ')
        }
        return ''
    }

    width: parent ? parent.width : 0
    topPadding: Theme.paddingMedium

    onContactChanged: {
        if (contact && contact.id !== 0) {
            contact.fetchConstituents()
        }
    }

    ListItem {
        id: highlightArea

        highlightedColor: "transparent"
        _showPress: false

        contentHeight: {
            var avatarHeight = avatarArea.height + labelColumn.height + blockedIndicator.height + Theme.paddingLarge
            return avatarHeight
        }

        width: parent.width
        acceptedButtons: Qt.LeftButton | Qt.RightButton

        PopupMenu {
            id: contextMenu

            PopupMenuItem {
                //: Menu option to copy the contact's display name shown in the contact card header
                //% "Copy name"
                text: qsTrId("components_contacts-me-copy_contact_name")
                visible: nameLabel.text.length > 0
                onClicked: Clipboard.text = nameLabel.text.replace("\n", " ")
            }

            PopupMenuItem {
                //: Menu option to copy the contact's extra details shown in the contact card header
                //% "Copy details"
                text: qsTrId("components_contacts-me-copy_contact_information")
                visible: extraDetailLabel.text.length > 0
                onClicked: Clipboard.text = extraDetailLabel.text
            }
        }

        onClicked: {
            if (mouse.button === Qt.RightButton) {
                contextMenu.open()
                return
            }

            contextMenu.open()
        }
        onPressAndHold: {
            contextMenu.open()
        }

        AvatarImage {
            id: avatarArea

            anchors {
                top: parent.top
                horizontalCenter: parent.horizontalCenter
            }
            contact: root.contact
            unsavedContact: root.unsavedContact
            highlighted: highlightArea.highlighted
        }

        Column {
            id: labelColumn

            spacing: Theme.paddingSmall
            width: parent.width - 2 * Theme.paddingLarge
            anchors {
                top: avatarArea.bottom
                topMargin: Theme.paddingMedium
                horizontalCenter: parent.horizontalCenter
            }

            Label {
                id: nameLabel
                width: parent.width
                font {
                    family: Theme.fontFamilyHeading
                    pixelSize: Theme.fontSizeExtraLarge
                }
                color: highlightArea.highlighted ? Theme.highlightColor : Theme.primaryColor

                horizontalAlignment: Text.AlignHCenter
                wrapMode: Text.Wrap
                maximumLineCount: 3
                text: getNameText()
            }


            Label {
                id: extraDetailLabel
                width: parent.width
                height: text.length > 0 ? implicitHeight : 0
                font {
                    family: Theme.fontFamilyHeading
                    pixelSize: Theme.fontSizeMedium
                }
                color: highlightArea.highlighted ? Theme.secondaryHighlightColor : Theme.secondaryColor
                horizontalAlignment: Text.AlignHCenter
                wrapMode: Text.Wrap
                maximumLineCount: 30
                text: getDetailText()
            }
        }
        Row {
            id: blockedIndicator

            anchors {
                top: labelColumn.bottom
                topMargin: Theme.paddingLarge
                horizontalCenter: parent.horizontalCenter
            }
            height: root.contactIsBlocked ? Theme.dp(80) : 0

            visible: root.contactIsBlocked
            spacing: Theme.paddingSmall
            Icon {
                anchors.verticalCenter: parent.verticalCenter
                source: "image://theme/icon-splus-blocked"
                color: Theme.highlightColor
            }
            Label {
                anchors.verticalCenter: parent.verticalCenter
                text: qsTrId("components_contacts-block_notification_message")
                color: Theme.highlightColor
            }
        }
    }
}
