/*
 * SPDX-FileCopyrightText: 2019 Jolla Ltd
 * SPDX-FileCopyrightText: 2019-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.6
import Sailfish.Silica 1.0
import Sailfish.Contacts 1.0 as SailfishContacts
import Sailfish.Silica.private 1.0 as Private
import org.nemomobile.contacts 1.0
import "detaileditors"

/**
 * Main editor page that contains sections for each type of contact detail.
 * These sections and their names are populated on this page, but each section
 * populates its own data from the contact object that is passed from here to
 * them.
 */
Dialog {
    id: root

    property var peopleModel
    property Person subject
    property var focusField: ({})

    property var _originalContactData
    property Person _contact: subject && subject.complete && !_readOnly ? subject : null
    property var _peopleModel: peopleModel || SailfishContacts.ContactModelCache.unfilteredModel()
    property var _editors: [name, company, phone, email, note, address, website]
    readonly property bool _readOnly: !subject
                                      || !subject.complete
                                      || !ContactsUtil.isWritableContact(subject)

    readonly property bool isTablet: Private.LayoutClassProvider.currentDisplayClass & Private.DisplayLayoutClass.Tablet

    function findNextItemInFocusChain(fromItem, editor) {
        // When focused on last editor, don't loop focus back to the first editor.
        var nextItem = fromItem.nextItemInFocusChain(true)
        return nextItem === name.detailEditors.itemAt(0) ? null : nextItem
    }

    function hasDetailedContent() {
        if (date.hasContent && !date.testDate) {
            return false
        }

        for (var i = 0; i < _editors.length; i++) {
            if (_editors[i].hasContent) {
                return true
            }
        }

        return false
    }

    function hasCorrectPhoneNumber() {
        if (root._contact.blocked && phone.detailModel.count > 1) {
            for (var i = 0; i < phone.detailModel.count; i++) {
                var phoneNumber = phone.detailModel.get(i).value.trim()
                if (phoneNumber !== "" && PeopleModelUtils.isShortNumber(phoneNumber)) {
                    return false
                }
            }
        }

        return true
    }

    function resetToInitialState() {
        // Reset the contact back to its initial state
        _contact.setContactData(_originalContactData)
        if (_contact.id != 0) {
            _contact.resetContactData()
        }
    }

    InfoLabel {
        anchors.centerIn: parent

        visible: !flick.visible
        //% "This contact cannot be modified"
        text: qsTrId("components_contacts-la-contact_cannot_be_modified")
    }

    canAccept: !_readOnly && hasDetailedContent() && addressBookCombo.currentIndex >= 0 && date.isValideDate

    onAcceptBlocked: {
        // Name has not been entered. Focus the field to indicate it is required.
        root.focusField = { detailType: "name", detailIndex: 0 }
    }

    onDone: {
        if (result === DialogResult.Accepted) {
            if (!hasCorrectPhoneNumber()) {
                //% "Saving is not possible"
                Notices.show(qsTrId("components_contacts-cannot_save"), Notice.Short, Notice.Bottom)

                resetToInitialState()
                return
            }

            // Break the binding so that if contact state changes while saving and this affects the
            // id and readOnly state, the editor will not suddenly disappear.
            flick.visible = true

            flick.save()

            if (!_peopleModel.savePerson(_contact)) {
                console.log("Contact save failed!")
            } else {
                addressBookCombo.saveDefaultAddressBook()
            }
        } else {
            resetToInitialState()
        }

        // Close vkb
        focus = true
    }

    on_ContactChanged: {
        if (_contact) {
            _originalContactData = _contact.contactData()
        }
    }

    SilicaFlickable {
        id: flick

        function save() {
            for (var i = 0; i < _editors.length; ++i) {
                _editors[i].aboutToSave()
            }

            if (date.hasContent) {
                date.aboutToSave()
            }

            nicknames.copyMultiTypeDetailChanges(root._contact, nicknames.propertyAccessor)
        }

        anchors.fill: parent
        contentHeight: editorsColumn.height + header.height
        contentWidth: parent.width
        visible: !root._readOnly
        bottomMargin: Theme.paddingLarge * 4 + Theme.paddingMedium

        // Delay loading so that the editor doesn't appear abruptly when pushing immediately from
        // the constituent picker.
        NumberAnimation on opacity {
            from: 0
            to: 1
        }

        SailfishContacts.ContactDetailSuggestions {
            id: fieldSuggestions

            property alias inputItem: autoFill.inputItem
        }

        Private.AutoFill {
            id: autoFill

            suggestions: fieldSuggestions.suggestions
            canRemove: false
            inputItem: null
        }

        DialogHeader {
            id: header

            dialog: root
            //% "Save"
            acceptText: qsTrId("components_contacts-ce_accept_text")
        }

        Column {
            id: editorsColumn

            anchors {
                top: header.bottom
                horizontalCenter: parent.horizontalCenter
            }

            width: root.isLandscape && root.isTablet ? Screen.width : root.width - 2 * Theme.paddingMedium
            spacing: Theme.paddingMedium

            Item {
                id: avatarItem

                width: parent.width
                height: avatar.height + avatarLabel.height + Theme.paddingLarge

                ContactAvatar {
                    id: avatar

                    anchors {
                        horizontalCenter: parent.horizontalCenter
                    }
                    contact: root._contact
                }

                BackgroundItem {
                    id: avatarLabelItem

                    anchors {
                        top: avatar.bottom
                        topMargin: Theme.paddingMedium
                        horizontalCenter: avatarItem.horizontalCenter
                    }

                    height: avatarLabel.height
                    width: avatarItem.width

                    onClicked: {
                        avatar._changeAvatar()
                    }

                    Label {
                        id: avatarLabel

                        anchors.horizontalCenter: parent.horizontalCenter
                        font.pixelSize: Theme.fontSizeSmall
                        color: avatarLabelItem.down ? Theme.primaryColor : Theme.highlightColor
                        text: !!root._contact && root._contact.avatarUrl != ""
                              //% "Edit photo"
                              ? qsTrId("contacts-bt-contact_edit_photo")
                              //% "Add photo"
                              : qsTrId("contacts-bt-contact_add_photo")
                    }
                }
            }

            ExpanderItem {
                id: baseContainer

                showExpandButton: false

                NameEditor {
                    id: name

                    expanded: baseContainer.expanded
                    showBackground: false
                    width: editorsColumn.width
                    contact: root._contact
                    peopleModel: root._peopleModel
                    nicknameModel: nicknames
                    initialFocusIndex: focusField.detailType === "name" ? focusField.detailIndex : -1
                }

                PhoneEditor {
                    id: phone

                    showExpandButton: false
                    showBackground: false
                    width: editorsColumn.width
                    contact: root._contact
                    peopleModel: root._peopleModel
                    initialFocusIndex: focusField.detailType === "phone" ? focusField.detailIndex : -1
                }

                EmailEditor {
                    id: email

                    showExpandButton: false
                    showBackground: false
                    width: editorsColumn.width
                    contact: root._contact
                    peopleModel: root._peopleModel
                    initialFocusIndex: focusField.detailType === "email" ? focusField.detailIndex : -1
                }
            }

            CompanyEditor {
                id: company

                _contentContainer.anchors.topMargin: Theme.paddingLarge

                showBackground: true
                width: editorsColumn.width
                contact: root._contact
                peopleModel: root._peopleModel
                suggestions: fieldSuggestions
                initialFocusIndex: focusField.detailType === "company" ? focusField.detailIndex : -1
            }

            AddressEditor {
                id: address

                _titleIconContainer.anchors.topMargin: Theme.paddingLarge

                showBackground: true
                width: editorsColumn.width
                contact: root._contact
                peopleModel: root._peopleModel
                initialFocusIndex: focusField.detailType === "address" ? focusField.detailIndex : -1
            }
            ExpanderItem {
                id: additionalInfoContainer

                showExpandButton: false
                NoteEditor {
                    id: note

                    showBackground: false
                    width: editorsColumn.width
                    contact: root._contact
                    peopleModel: root._peopleModel
                    initialFocusIndex: focusField.detailType === "note" ? focusField.detailIndex : -1
                }

                DateEditor {
                    id: date

                    showBackground: false
                    width: editorsColumn.width
                    contact: root._contact
                    peopleModel: root._peopleModel
                    // initialFocusIndex not set, date field cannot be focused
                }

                WebsiteEditor {
                    id: website

                    showBackground: false
                    width: editorsColumn.width
                    contact: root._contact
                    peopleModel: root._peopleModel
                    initialFocusIndex: focusField.detailType === "website" ? focusField.detailIndex : -1
                }
            }

            ContactAddressBookComboBox {
                id: addressBookCombo

                contact: root._contact
                enabled: root._contact != null && root._contact.id === 0
            }
        }

        VerticalScrollDecorator { }
    }

    NicknameDetailModel {
        id: nicknames

        contact: root._contact
    }

    Component.onCompleted: {
        if (root.focusField.detailType == null) {
            root.focusField = { detailType: "name", detailIndex: 0 }
        }
    }
}
