/*
 * Copyright (c) 2019 - 2023 Open Mobile Platform LLC.
 * Copyright (c) 2013 - 2019 Jolla Pty Ltd.
 *
 * License: Proprietary
*/

import QtQuick 2.6
import Sailfish.Silica 1.0
import Sailfish.Contacts 1.0
import org.nemomobile.contacts 1.0

PullDownMenu {
    id: root

    property var page
    property var peopleModel
    property var contact

    property bool isBlockedContact: false

    readonly property bool _contactValid: !!contact && contact.id
    property var _peopleModel: peopleModel || ContactModelCache.unfilteredModel()

    signal unsavedContactLinkRequested()

    function triggerLinkAction() {
        linkMenuItem.clicked()
    }

    function _vCardName(person) {
        return person.displayLabel.replace(/\s/g, '') + '.vcf'
    }

    function _deleteContact() {
        var _contact = contact
        var cache = ContactModelCache
        cache._deletingContactId = contact.id
        var _page = pageStack.previousPage(page)
        pageStack.pop()
        var remorse = Remorse.popupAction(
                    _page,
                    //: Deleted contact, providing a way to undo for 4 seconds
                    //% "Deleted contact"
                    qsTrId("components_contacts-la-deleted_contact"),
                    function() {
                        cache.deleteContact(_contact)
                        cache._deletingContactId = -1
                    })
        remorse.canceled.connect(function () { cache._deletingContactId = -1 })
    }

    ContactDeleteMenuItem {
        contact: root.contact
        peopleModel: root._peopleModel
        visible: root._contactValid

        onClicked: root._deleteContact()
    }

    MenuItem {
        id: blockContactMenuItem

        //% "Unblock contact"
        text: isBlockedContact ? qsTrId("components_contacts-unblock_contact")
                                //% "Block contact"
                               : qsTrId("components_contacts-block_contact")
        onClicked: {
            var blockedName = root.contact.primaryName + " " + root.contact.secondaryName
            var contact = root.contact && root.contact.id ? root.contact : ''
            var editorProperties = {
                "blockedName": blockedName,
                "contact": contact,
                "phoneNumber": root.contact.phoneDetails[0].number
            }
            pageStack.animatorPush("Sailfish.Contacts.BlockContactDialog", editorProperties, PageStackAction.Animated)
        }
    }

    MenuItem {
        id: linkMenuItem

        objectName: "linkContactMenuItem"
        //: Manage links (associated contacts) for this contact
        //% "Link with contact"
        text: qsTrId("components_contacts-me-link")
        onClicked: {
            if (root._contactValid) {
                pageStack.animatorPush(Qt.resolvedUrl("ContactLinksPage.qml"),
                                       { "person": root.contact, "peopleModel": root._peopleModel } )
            } else {
                root.unsavedContactLinkRequested()
            }
        }
    }

    MenuItem {
        objectName: "shareContactMenuItem"
        //: Share contact via Bluetooth, Email, etc.
        //% "Share"
        text: qsTrId("components_contacts-me-share")
        visible: root._contactValid

        onClicked: {
            var content = {
                "data": root.contact.vCard(),
                "name": root._vCardName(root.contact),
                "type": "text/vcard",
                "icon": root.contact.avatarPath.toString()
            }
            _shareAction.resources = [content]
            _shareAction.trigger()
        }
    }

    property var _shareAction: ContactShareAction {}

    ContactEditMenuItem {
        contact: root.contact
        peopleModel: root._peopleModel
        visible: root._contactValid

        onClicked: {
            ContactsUtil.editContact(root.contact, root._peopleModel, pageStack)
        }
    }

}
