/*
 * SPDX-FileCopyrightText: 2013-2019 Jolla Pty Ltd
 * SPDX-FileCopyrightText: 2020-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.6
import Aurora.Controls 1.0
import Sailfish.Silica 1.0
import Sailfish.Silica.private 1.0
import Sailfish.Contacts 1.0 as SailfishContacts
import org.nemomobile.contacts 1.0
import Aurora.Controls.private 1.0
import Aurora.Applications.Internal 1.0 as AppInternal

Page {
    id: root

    readonly property bool isDeletable: !!root.contact
                                        && root.contact.id !== 0
                                        && SailfishContacts.ContactsUtil.allConstituentsDeletable(root.contact, _peopleModel)

    readonly property bool isEditable: SailfishContacts.ContactsUtil.isWritableContact(root.contact.constituents.length > 1
                                                                                       ? root.contact
                                                                                       : _peopleModel.personById(root.contact.constituents[0]))

    readonly property bool isPreview: !root.contact.addressBook.isAggregate && !isEditable
    readonly property bool isTablet: LayoutClassProvider.currentDisplayClass & DisplayLayoutClass.Tablet

    readonly property bool contactCardReady: !root.contact
                                             || (_unsavedContact && _unsavedContact.busy)
                                             || (contactCardLoader.status !== Loader.Ready)

    property alias contactCard: contactCardLoader.item

    property QtObject contact
    property bool promptLinkImmediately

    property string activeDetail

    property var _unsavedContact
    property var _peopleModel: SailfishContacts.ContactModelCache.unfilteredModel()
    property var _shareAction: ContactShareAction {}

    function showError(errorText) {
        if (errorText) {
            errorLabel.text = errorText
        }
    }

    function _vCardName(person) {
        return person.displayLabel.replace(/\s/g, '') + '.vcf'
    }

    function anyConstituentWritable() {
        if (!contact.addressBook.isAggregate) {
            return false
        }
        if (contact.constituents.length === 0) {
            findConstituents.target = contact
            contact.fetchConstituents()
            return false
        }
        var constituents = contact.constituents
        for (var i = 0; i < constituents.length; ++i) {
            var constituentContact = peopleModel.personById(constituents[i])

            // If any constituent is editable, then the aggregate is considered editable.
            if (constituentContact && !constituentContact.addressBook.readOnly) {
                return true
            }
        }
        return false
    }

    function _deleteContact(contactId, isAggregated) {
        const cache = SailfishContacts.ContactModelCache
        cache._deletingContactId = !!contactId ? contactId : root.contact.id

        const obj = pageStack.animatorPush(Qt.resolvedUrl("RemoveContactsDialog.qml"), {
                                               "contactsToRemove": [cache._deletingContactId],
                                               "person": root.contact,
                                               "acceptDestination": pageStack.previousPage(pageStack.currentPage),
                                               "acceptDestinationAction": PageStackAction.Pop
                                           })

        obj.pageCompleted.connect(function(page) {
            var contactToDelete = root.contact

            if (isAggregated) {
                var constituent = root._peopleModel.personById(contactId)

                if (constituent) {
                    root.contact.disaggregateFrom(constituent)
                    contactToDelete = constituent
                }
            }

            page.accepted.connect(function() {
                SailfishContacts.ContactModelCache.deleteContact(contactToDelete, {
                     successCallback: function() {
                         //% "Contact removed"
                         Notices.show(qsTrId("components_contacts-me-removed_contact"), Notice.Short, Notice.Bottom)
                     },
                     errorCallback: function(errorText) {
                         Notices.show(errorText, Notice.Short, Notice.Bottom)
                     }
                 })

                SailfishContacts.ContactModelCache._deletingContactId = -1
            })
        })
    }

    Binding on activeDetail {
        when: contactCardLoader.item
        value: contactCardLoader.item.activeDetail
    }

    Connections {
        id: findConstituents

        target: null

        onConstituentsChanged: {
            target = null
        }
    }

    onContactChanged: {
        if (contact && contact.id === 0 && !_unsavedContact) {
            _unsavedContact = unsavedContactComponent.createObject(root)
        }

        if (contact) {
            contactCardLoader.setSource(Qt.resolvedUrl("ContactCard.qml"),
                                        {
                                            "contact": root.contact,
                                            "unsavedContact": _unsavedContact && _unsavedContact.contact && _unsavedContact.contact.id === 0
                                        })
        }
    }

    Component {
        id: unsavedContactComponent

        UnsavedContactCardHandler {
            contactCardPage: root
            contact: !!root.contact && root.contact.id === 0 ? root.contact : null
            peopleModel: SailfishContacts.ContactModelCache.unfilteredModel()

            // If the unsaved contact is resolved or saved, update the card to show the resolved
            // or saved contact.
            onResolvedToContact: contactCard.contact = contact
            onSavedAsContact: contactCard.contact = contact

            // If the contact becomes aggregated into another contact, show that contact instead.
            onAggregatedIntoContact: contactCard.contact = contact

            onActiveDetailChanged: contactCard.activeDetail = activeDetail
            onError: root.showError(errorText)
        }

    }

    ContactFavoriteModifier {
        id: favoriteModifier

        peopleModel: SailfishContacts.ContactModelCache.unfilteredModel()
    }

    AppBar {
        id: appBar

        divider: false

        AppBarSpacer {}

        AppBarButton {
            visible: !root.isPreview
            opacity: !isEditable ? Theme.opacityHigh : 1.0
            icon.source: "image://theme/icon-splus-edit"

            onClicked: {
                if (!isEditable) {
                    SailfishContacts.ContactsUtil.showNotice(SailfishContacts.ContactsUtil.contactUnableActionsNames[SailfishContacts.ContactsUtil.ActionsWithConstContact.Edit],
                                                             Notice.Short,
                                                             Notice.Bottom,
                                                             0,
                                                             -Theme.dp(142))
                    return
                }

                SailfishContacts.ContactsUtil.editContact(root.contact, root._peopleModel, pageStack)
            }
        }

        AppBarButton {
            visible: !root.isPreview
            icon.source: ((favoriteModifier.lastStatusValid ? favoriteModifier.lastStatus : root.contact.favorite)
                          ? "image://theme/icon-splus-favorite-selected"
                          : "image://theme/icon-splus-favorite")
            enabled: !contactCard.isBlockedContact && !!root.contact && root.contact.id !== 0 && root.contact.complete

            onClicked: {
                if (!!root.contact)
                    favoriteModifier.setFavoriteStatus([root.contact], !root.contact.favorite)
            }
        }

        AppBarButton {
            id: moreButton

            icon.source: "image://theme/icon-splus-more"

            onClicked: popUpMenu.open()
        }
    }

    Rectangle {
        anchors.top: appBar.bottom
        height: Theme.dp(1)
        width: appBar.width
        color: Theme.rgba(Theme.primaryColor, 0.2)
    }

    PopupMenu {
        id: popUpMenu

        parent: moreButton

        PopupMenuItem {
            //: Deletes contact
            //% "Delete contact"
            text: qsTrId("components_contacts-me-delete_contact")
            visible: !_unsavedContact
                     && !root.isPreview
            opacity: isEditable ? 1.0 : Theme.opacityHigh
            icon.source: "image://theme/icon-splus-delete"

            onClicked: {
                if (isEditable) {
                    var aggregateContact = SailfishContacts.ContactsUtil.ensureContactComplete(root.contact,
                                                                                               root._peopleModel)
                    if (!isDeletable) {
                        var obj = pageStack.animatorPush("Sailfish.Contacts.ConstituentPicker",
                                                         {
                                                             "aggregateContact": aggregateContact,
                                                             "peopleModel": root._peopleModel
                                                         })
                        obj.pageCompleted.connect(function(page) {
                            page.constituentClicked.connect(function(constituentId) {
                                var constituent = root._peopleModel.personById(constituentId)
                                if (constituent) {
                                    root.contact.disaggregateFrom(constituent)
                                    root._deleteContact(constituentId, !isDeletable)
                                }
                            })
                        })
                    } else {
                        root._deleteContact()
                    }
                } else {
                    const key = SailfishContacts.ContactsUtil.ActionsWithConstContact.Delete
                    const noticeText = SailfishContacts.ContactsUtil.contactUnableActionsNames[key]
                    SailfishContacts.ContactsUtil.showNotice(noticeText,
                                                             Notice.Short,
                                                             Notice.Bottom,
                                                             0,
                                                             -Theme.dp(142))
                }
            }
        }

        PopupMenuItem {
            //% "Unblock contact"
            text: contactCard.isBlockedContact ? qsTrId("components_contacts-unblock_contact")
                                                 //% "Block contact"
                                               : qsTrId("components_contacts-block_contact")
            visible: !root.isPreview
            icon.source: "image://theme/icon-splus-blocked"

            onClicked: {
                var blockedName = root.contact.primaryName + " " + root.contact.secondaryName
                var contact = root.contact && root.contact.id ? root.contact : ''
                var editorProperties = {
                    "blockedName": blockedName,
                    "contact": contact,
                    "phoneNumber": root.contact.phoneDetails[0].number
                }
                pageStack.animatorPush("Sailfish.Contacts.BlockContactDialog", editorProperties, PageStackAction.Animated)
            }
        }

        PopupMenuItem {
            //: Manage links (associated contacts) for this contact
            //% "Link with contact"
            text: qsTrId("components_contacts-me-link")
            visible: !_unsavedContact
                     && !root.isPreview
            enabled: isEditable
            icon.source: "image://theme/icon-splus-link"

            onClicked: {
                if (!!root.contact && root.contact.id !== 0 && root.contact.complete) {
                    pageStack.animatorPush(Qt.resolvedUrl("ContactLinksPage.qml"),
                                           { "person": root.contact, "peopleModel": root._peopleModel } )
                } else {
                    if (!!_unsavedContact) {
                        _unsavedContact.selectContactToLink(PageStackAction.Animated)
                    }
                }
            }
        }

        PopupMenuItem {
            //: Share contact via Bluetooth, Email, etc.
            //% "Share"
            text: qsTrId("components_contacts-me-share")
            visible: !_unsavedContact
            icon.source: "image://theme/icon-splus-share"

            onClicked: {
                var content = {
                    "data": root.contact.vCard(),
                    "name": root._vCardName(root.contact),
                    "type": "text/vcard",
                    "icon": root.contact.avatarPath.toString()
                }
                _shareAction.resources = [content]
                _shareAction.trigger()
            }
        }

        PopupMenuItem {
            objectName: "saveContactMenuItem"

            //: Save contact
            //% "Save"
            text: qsTrId("components_contacts-me-save")
            visible: !!_unsavedContact
                     && !!_unsavedContact.contact
            icon.source: "image://theme/icon-splus-add"

            onClicked: {
                ContactsUtil.editContact(_unsavedContact.contact,
                                         SailfishContacts.ContactModelCache.unfilteredModel(),
                                         pageStack,
                                         {
                                             "acceptDestination": root,
                                             "acceptDestinationAction": PageStackAction.Pop,
                                         })
            }
        }

        Component.onCompleted: {
            popUpMenu.preferredWidth = AppInternal.UiUtils.getPopupMenuPreferredWidth({
                popupMenu: popUpMenu,
                minimumWidth: AppInternal.UiUtils.isLargeScreen
                                ? popUpMenu.preferredWidth
                                : root.width - Theme.dp(68),
                maximumWidth: root.width
            })
        }
    }

    TextMetrics {
        id: tm

        font.pixelSize: Theme.fontSizeMedium // from PopupMenuBaseItem
    }

    Label {
        id: errorLabel

        anchors.centerIn: parent
        width: parent.width - Theme.horizontalPageMargin*2
        x: Theme.horizontalPageMargin
        //% "Contact not found"
        text: qsTrId("components_contacts-la-contact_not_found")
        wrapMode: Text.Wrap
        horizontalAlignment: Text.AlignHCenter
        color: Theme.highlightColor
        visible: !root.contact && !root.contactCardReady
        font.pixelSize: Theme.fontSizeLarge
    }

    Loader {
        id: contactCardLoader

        asynchronous: true
        visible: !errorLabel.visible
        width: root.isLandscape && root.isTablet ? Screen.width : root.width

        anchors {
            horizontalCenter: parent.horizontalCenter
            top: appBar.bottom
            bottom: parent.bottom
        }
    }

    Component.onCompleted: {
        if (contact && contact.id === 0 && !!_unsavedContact) {
            _unsavedContact = unsavedContactComponent.createObject(root)
        }
        if (promptLinkImmediately) {
            if (!!contact && contact.id) {
                pageStack.animatorPush(Qt.resolvedUrl("ContactLinksPage.qml"),
                                       { "person": root.contact, "peopleModel": root._peopleModel } )
            } else {
                if (_unsavedContact != null) {
                    _unsavedContact.selectContactToLink(PageStackAction.Animated)
                }
            }
        }

        if (root.contact) {
            SailfishContacts.ContactCardUtils.instance()
            root.contact.fetchConstituents()
            contactCardLoader.setSource(Qt.resolvedUrl("ContactCard.qml"),
                                        {
                                            "contact": root.contact,
                                            "unsavedContact": _unsavedContact && _unsavedContact.contact && _unsavedContact.contact.id === 0
                                        })
        }
    }
}
