/**
 * SPDX-FileCopyrightText: 2013-2019 Jolla Pty Ltd
 * SPDX-FileCopyrightText: 2019-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.6
import QtQuick.Layouts 1.1
import Aurora.Controls 1.0
import Sailfish.Silica 1.0
import Sailfish.Silica.private 1.0
import Sailfish.Telephony 1.0
import Sailfish.Contacts 1.0 as SailfishContacts
import Sailfish.AccessControl 1.0
import Aurora.Controls.private 1.0
import MeeGo.QOfono 0.2
import MeeGo.Connman 0.2
import org.nemomobile.contacts 1.0
import Aurora.Applications.Internal 1.0 as AppInternal

import Nemo.DBus 2.0
import "contactcard/contactcardmodelfactory.js" as ModelFactory
import "contactcard"

SilicaListView {
    id: root

    clip: true

    property Person contact
    property string activeDetail
    readonly property bool actionPermitted: AccessControl.hasGroup(AccessControl.RealUid, "sailfish-phone")
                                            || AccessControl.hasGroup(AccessControl.RealUid, "sailfish-messages")
    property bool hidePhoneActions: SailfishContacts.ContactCardUtils.cellular1Status.modemPath.length === 0 && SailfishContacts.ContactCardUtils.cellular2Status.modemPath.length === 0 || !actionPermitted
    property bool disablePhoneActions: !SailfishContacts.ContactCardUtils.cellular1Status.registered && !SailfishContacts.ContactCardUtils.cellular2Status.registered && !SailfishContacts.ContactCardUtils.network.flightMode
    property bool isBlockedContact: SailfishContacts.ContactModelCache.isBlockedContact(contact) || SailfishContacts.ContactModelCache.isBlockedContact(activeDetail)

    property bool unsavedContact

    readonly property bool isReady: (contact && contact.complete)

    property ListModel detailsModel: ListModel { }

    property bool lastFavoriteStatus
    property bool favoriteStateChanged: false

    property var _constituent

    function refreshDetails() {
        if (!favoriteStateChanged) {
            lastFavoriteStatus = contact.favorite
            favoriteStateChanged = true
        } else if (lastFavoriteStatus !== contact.favorite) {
            lastFavoriteStatus = contact.favorite
            return
        }

        detailsModel.clear()

        SailfishContacts.ContactsUtil.init()
        ModelFactory.init(SailfishContacts.ContactsUtil)

        ModelFactory.getContactCardDetailsModel(detailsModel, contact)
    }

    function _asyncRefresh() {
        if (contact.complete) {
            contact.completeChanged.disconnect(_asyncRefresh)
            refreshDetails()
        }
    }

    function _editDetail() {
        if (!detailContextMenu.detailsOriginId) {
            return
        }

        if (!_constituent) {
            _constituent = SailfishContacts.ContactModelCache.unfilteredModel().personById(detailContextMenu.detailsOriginId)
        }

        if (!_constituent) {
            console.warn("Cannot find contact for id:", detailContextMenu.detailsOriginId)
            return
        }

        if (_constituent.complete) {
            _doEditDetail()
        } else {
            _constituent.completeChanged.connect(_doEditDetail)
        }
    }

    function _doEditDetail() {
        _constituent.completeChanged.disconnect(_doEditDetail)

        var extraProperties = {
            "focusField": {
                "detailType": detailContextMenu.detailsType,
                "detailIndex": detailContextMenu.detailsIndex
            }
        }

        ContactsUtil.editContact(_constituent,
                                 SailfishContacts.ContactModelCache.unfilteredModel(),
                                 pageStack,
                                 extraProperties)
    }

    onContactChanged: {
        if (contact) {
            root.favoriteStateChanged = false
            if (contact.complete) {
                contact.fetchConstituents()
                refreshDetails()
            } else {
                contact.completeChanged.connect(_asyncRefresh)
            }
        } else {
            detailsModel.clear()
        }
    }

    Connections {
        target: contact || null

        onDataChanged: {
            root.isBlockedContact = SailfishContacts.ContactModelCache.isBlockedContact(contact) || SailfishContacts.ContactModelCache.isBlockedContact(activeDetail)
            refreshDetails()
        }

        onConstituentsChanged: refreshDetails()
    }

    height: parent.height

    spacing: Theme.paddingMedium

    model: contact.complete ? detailsModel : 0

    quickScroll: true

    header: ContactHeader {
        id: header

        contact: root.contact
        simManager: SailfishContacts.ContactCardUtils.simManager
        contactIsBlocked: root.isBlockedContact
        unsavedContact: root.unsavedContact
    }

    footer: Item {
        width: ListView.view.width
        height: Theme.dp(46)
    }

    delegate: Item {
        visible: !sectionLoader.item || (sectionLoader.item && sectionLoader.item.enabled)
        width: ListView.view.width
        height: (sectionLoader.item
                 && !sectionLoader.item.enabled) ? 0
                                                 : (sectionLoader.height || skeleton.height)
                                                   ? Math.max(sectionLoader.height, skeleton.height)
                                                   : ListView.view.height

        Loader {
            id: sectionLoader

            property Person contact: root.contact
            property PopupMenu contextMenu: detailContextMenu
            property var simMenu: _simMenu
            property var modelData: model

            anchors {
                left: parent.left
                leftMargin: Theme.paddingMedium
                right: parent.right
                rightMargin: Theme.paddingMedium
            }

            visible: status === Loader.Ready
            asynchronous: true

            source: {
                switch (model.sectionType) {
                case "contactData":
                    return Qt.resolvedUrl("contactcard/ContactDetailsSection.qml")
                case "addressData":
                    return Qt.resolvedUrl("contactcard/AddressSection.qml")
                case "additionalData":
                    return Qt.resolvedUrl("contactcard/AdditionalSection.qml")
                case "activityData":
                    return Qt.resolvedUrl("contactcard/EventsSection.qml")
                case "storageData":
                    return Qt.resolvedUrl("contactcard/StorageSection.qml")
                case "aggregateData":
                    return Qt.resolvedUrl("contactcard/AggregateSection.qml")
                }

                return ""
            }
        }

        ContactCardSection {
            id: skeleton

            property bool showSkeleton: !activitySkeletonDelayTimer.running

            anchors {
                left: parent.left
                leftMargin: Theme.paddingMedium
                right: parent.right
                rightMargin: Theme.paddingMedium
            }

            visible: !sectionLoader.visible && showSkeleton

            header: {
                switch (model.sectionType) {
                case "contactData":
                    return qsTrId("components_contacts-la-contacts")
                case "addressData":
                    return qsTrId("components_contacts-la-address")
                case "additionalData":
                    return qsTrId("components_contacts-la-additional")
                case "activityData":
                    return qsTrId("components_contacts-la-activity")
                case "storageData":
                    return qsTrId("components_contacts-la-storage")
                case "aggregateData":
                    return qsTrId("components_contacts-la-aggregate")
                }
                return ""
            }

            fields: [ Item {
                width: parent.width
                height: Theme.dp(100)

                Skeleton {
                    id: mainIconSkeleton

                    y: Theme.dp(15)
                    radius: Theme.dp(6)
                    width: Theme.dp(48)
                    height: Theme.dp(48)
                }

                Skeleton {
                    id: mainLabelSkeleton

                    anchors {
                        left: mainIconSkeleton.right
                        leftMargin: Theme.paddingLarge
                        right: secondIconSkeleton.left
                        rightMargin: Theme.dp(64)
                    }

                    radius: Theme.dp(6)
                    y: Theme.paddingMedium
                    height: Theme.dp(36)
                }

                Skeleton {
                    anchors {
                        top: mainLabelSkeleton.bottom
                        topMargin: Theme.paddingMedium
                        left: mainLabelSkeleton.left
                        right: mainLabelSkeleton.right
                        rightMargin: Theme.dp(204)
                    }

                    radius: Theme.dp(6)
                    height: Theme.dp(28)
                }

                Skeleton {
                    id: secondIconSkeleton

                    y: Theme.dp(15)
                    radius: Theme.dp(6)
                    width: Theme.dp(48)
                    height: Theme.dp(48)
                    anchors.right: parent.right
                }
            } ]

            Timer {
                id: activitySkeletonDelayTimer

                running: model.sectionType === "activityData"
                repeat: false
                interval: 150
            }
        }
    }

    PopupMenu {
        id: detailContextMenu

        property string clipboardText
        property var detailsOriginId
        property var detailsType
        property var detailsIndex

        PopupMenuItem {
            //% "Copy to clipboard"
            text: qsTrId("components_contacts-action-copy_to_clipboard")

            onClicked: {
                Clipboard.text = detailContextMenu.clipboardText
            }
        }

        PopupMenuItem {
            //: Edit a particular detail value, e.g. phone number or email address
            //% "Edit"
            text: qsTrId("components_contacts-me-edit_detail")

            onClicked: {
                _editDetail()
            }
        }

        Component.onCompleted: {
            detailContextMenu.preferredWidth = AppInternal.UiUtils.getPopupMenuPreferredWidth({
                popupMenu: detailContextMenu,
                minimumWidth: AppInternal.UiUtils.isLargeScreen
                                ? detailContextMenu.preferredWidth
                                : root.width,
                maximumWidth: root.width
            })
        }
    }

    AppInternal.LazyPopupMenu {
        id: _simMenu

        function openOnCurrentItem(currentItem, number) {
            if (Telephony.voiceSimUsageMode === Telephony.AlwaysAskSim &&
                    SailfishContacts.ContactCardUtils.simManager.availableModemCount > 0) {
                _simMenu.promise({
                                     resolve: function(instance) {
                                         instance.dialNumber = number
                                         instance.open(currentItem)
                                     }
                                 })
            } else {
                voicecall.dial(number)
            }
        }

        parent: detailContextMenu
        sourceComponent: PopupMenu {
            id: _simPopupMenu

            property string dialNumber: ""

            preferredWidth: detailContextMenu.preferredWidth

            SimPicker {
                id: simPicker

                onSimSelected: {
                    voicecall.dialViaModem(modemPath, _simPopupMenu.dialNumber)
                    _simPopupMenu.close()
                    _simPopupMenu.dialNumber = ""
                }
            }
        }
    }

    DBusInterface {
        id: voicecall

        service: "com.jolla.voicecall.ui"
        path: "/"
        iface: "com.jolla.voicecall.ui"

        function dial(number) {
            call('dial', number)
        }

        function dialViaModem(modemPath, number) {
            call('dialViaModem', [ modemPath, number ])
        }
    }

    VerticalScrollDecorator {}
}
