/*
 * SPDX-FileCopyrightText: 2013-2020 Jolla Ltd
 * SPDX-FileCopyrightText: 2023-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.0
import Sailfish.Silica 1.0
import Sailfish.Contacts 1.0 as Contacts
import org.nemomobile.contacts 1.0

ContactItem {
    id: contactItem

    // Use getPerson() to access the person object so one isn't instantiated unnecessarily
    property int contactId
    property bool canDeleteContact: true
    property var peopleModel: ListView.view.model

    property var selectionModel
    property int selectionModelIndex: {
        if (selectionModel !== null && selectionModel.count > 0) {
            var ids = selectionModel.findContactId(contactId)
            if (ids.length > 0) return ids[0]
        }
        return -1 // count to retrigger on change.
    }
    property var disabledModel

    property var propertyPicker
    property var disabledItems: (disabledModel && disabledModel.count)
                                ? disabledModel.getDisabledItems(contactId) : null

    property Person person: {
        try {
            return getPerson()
        } catch (e) {
            return null
        }
    }

    signal contactClicked(var contact)
    signal contactPressAndHold(var contact)

    hidden: Contacts.ContactModelCache._deletingContactId === contactId
    highlighted: down || menuOpen || selectionModelIndex >= 0
    openMenuOnPressAndHold: false
    enabled: disabledItems ? model.phoneNumbers.length !== disabledItems.length : true
    opacity: enabled ? 1.0 : Theme.opacityLow
    acceptedButtons: Qt.LeftButton | Qt.RightButton

    firstText: Contacts.ContactModelCache.getPersonFirstText(contactItem.person)
    secondText: Contacts.ContactModelCache.getPersonSecondText(contactItem.person)

    function deleteContact() {
        if (contactId && peopleModel && getPerson()) {
            // Retrieve the person to delete; it will be no longer accessible if the
            // remorse function is triggered by delegate destruction
            // Similarly, cache a reference to the contact model cache.
            var contactModelCache = Contacts.ContactModelCache
            var person = getPerson()
            var item = remorseDelete(function () {
                contactModelCache.deleteContact(person)
            })
            item.rightMargin = Theme.paddingMedium
        }
    }

    function personObject() {
        return getPerson()
    }

    onClicked: {
        if (mouse.button === Qt.RightButton) {
            contactPressAndHold(getPerson())
            return
        }

        contactClicked(getPerson())
    }

    onPressAndHold: {
        contactPressAndHold(getPerson())
    }
}
