/*
 * SPDX-FileCopyrightText: Copyright 2019-2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 *
 * Copyright (c) 2020 Open Mobile Platform LLC.
 *
 * License: Proprietary
*/

import QtQuick 2.6
import Aurora.Controls 1.0
import Sailfish.Silica 1.0
import Sailfish.Contacts 1.0 as SailfishContacts
import Sailfish.Telephony 1.0
import org.nemomobile.contacts 1.0
import org.nemomobile.commhistory 1.0
import "contactcard/contactcardmodelfactory.js" as ModelFactory
import Aurora.Applications.Internal 1.0 as AppInternal

Page {
    id: root

    property var contact
    property string remoteUid: ""
    property bool hidePhoneActions
    property var modelFactory: ModelFactory
    property var simManager

    signal startPhoneCall(string number, string modemPath)
    signal startSms(string number)
    signal startInstantMessage(string localUid, string remoteUid)

    Component.onCompleted: {
        SailfishContacts.ContactsUtil.init()
        ModelFactory.init(SailfishContacts.ContactsUtil)
        ModelFactory.getContactCardDetailsModel(activityList.model, root.contact)
    }

    AppBar {
        id: appBar

        headerText: root.contact && root.contact.id !== 0
                    ? contact.displayLabel
                    : root.remoteUid
    }

    SilicaListView {
        id: activityList

        anchors {
            top: appBar.bottom
            topMargin: Theme.paddingSmall
            left: parent.left
            right: parent.right
            bottom: parent.bottom
        }
        spacing: Theme.paddingSmall
        opacity: 0
        Behavior on opacity { FadeAnimation {} }

        model: FormattingProxyModel {
            sourceModel: CommRecipientEventModel {
                contactId: root.contact ? root.contact.id : 0
                remoteUid: {
                    if (root.remoteUid) {
                        return root.remoteUid
                    }

                    if (root.contact) {
                        return SailfishContacts.ContactsUtil.firstPhoneNumber(root.contact)
                    }

                    return ""
                }

                onReadyChanged: {
                    if (ready) {
                        // Update opacity here instead of using a binding, as model ready value
                        // starts as true. When contact model loads its events, the change
                        // notify signal is emitted, with the value still being true.
                        activityList.opacity = 1
                    }
                }
            }
        }

        section {
            property: "endDate"

            delegate: SectionHeader {
                text: AppInternal.UiUtils.getDateSection(section, {
                    //% "Today"
                    today: qsTrId("components_contacts-today"),
                    //% "Yesterday"
                    yesterday: qsTrId("components_contacts-yesterday")
                })
                horizontalAlignment: Qt.AlignLeft
            }
        }

        delegate: ContactActivityDelegate {
            anchors {
                left: parent.left
                leftMargin: Theme.paddingMedium
                right: parent.right
                rightMargin: Theme.paddingMedium
            }

            simManager: root.simManager
            hidePhoneActions: root.hidePhoneActions
            modelFactory: root.modelFactory
            contact: root.contact

            onStartPhoneCall: root.startPhoneCall(model.remoteUid, modemPath)
            onStartSms: root.startSms(model.remoteUid)
            onStartInstantMessage: root.startInstantMessage(model.localUid, model.remoteUid)
        }

        VerticalScrollDecorator {}
    }
}
