/*
 * SPDX-FileCopyrightText: 2023-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.6
import Aurora.Controls 1.0
import Sailfish.Silica 1.0
import Sailfish.Policy 1.0
import Sailfish.Contacts 1.0
import Sailfish.Telephony 1.0
import org.nemomobile.contacts 1.0
import org.nemomobile.commhistory 1.0
import org.nemomobile.configuration 1.0
import Aurora.Applications.Internal 1.0 as AppInternal

ListItem {
    id: root

    property SimManager simManager
    property bool hidePhoneActions
    property var modelFactory
    property var contact
    property real leftMargin: Theme.paddingMedium + Theme.iconSizeMedium + Theme.paddingMedium
    property bool showYear

    property string showBackgroundPolicy: ContactsUtil.VisibilityPolicy.Automatic

    readonly property bool isPhone: model.localUid.indexOf('/ring/tel/') >= 0
    readonly property bool isMessage: model.eventType !== CommCallModel.CallEvent && model.eventType !== CommCallModel.VoicemailEvent
    readonly property int modemIndex: simManager.indexOfModemFromImsi(model.subscriberIdentity)

    readonly property bool forceShowBackground: root.pressed ||
                                                root.down ||
                                                contextMenu.openedOrInProgress ||
                                                simMenu.openedOrInProgress

    readonly property int formatType: timeFormatConfig.value === "12"
                                           ? Formatter.TimeValueTwelveHours
                                           : Formatter.TimeValueTwentyFourHours

    signal startPhoneCall(string modemPath)
    signal startSms()
    signal startInstantMessage()

    function getActionTitle() {
        if (model.direction === CommCallModel.Inbound) {
            return model.isMissedCall
            //% "Missed call"
                    ? qsTrId("components_contacts-la-missed_call")
                      //% "Incoming call"
                    : qsTrId("components_contacts-la-incoming_call")
        } else {
            //% "Outgoing call"
            return qsTrId("components_contacts-la-outgoing_call")
        }
    }

    function tryOpenContextMenu() {
        if (!isPhone || !hidePhoneActions) {
            contextMenu.open()
        }
    }

    anchors {
        left: parent.left
        right: parent.right
    }

    //--- internal properties follow

    highlighted: {
        switch (root.showBackgroundPolicy) {
        case ContactsUtil.VisibilityPolicy.Automatic:
        case ContactsUtil.VisibilityPolicy.AlwaysOn:
            return root.forceShowBackground
        default:
            return false
        }
    }

    highlightedColor: Theme.rgba(Theme.highlightColor, Theme.opacityFaint)

    _backgroundColor: {
        if (root.showBackgroundPolicy === ContactsUtil.VisibilityPolicy.AlwaysOn) {
            return root.highlightedColor
        }

        if (root.showBackgroundPolicy === ContactsUtil.VisibilityPolicy.Automatic
                && root.highlighted) {
            return root.highlightedColor
        }

        return "transparent"
    }

    _backgroundRadius: Theme.dp(16)
    height: contentHeight
    contentHeight: rowMain.height + Theme.paddingMedium
    acceptedButtons: Qt.LeftButton | Qt.RightButton
    contentItem.radius: Theme.dp(Theme.paddingSmall)

    onClicked: {
        if (mouse.button === Qt.RightButton) {
            root.tryOpenContextMenu()
            return
        }

        root.tryOpenContextMenu()
    }

    onPressAndHold: root.tryOpenContextMenu()

    ConfigurationValue {
        id: timeFormatConfig

        key: "/sailfish/i18n/lc_timeformat24h"
    }

    AppInternal.LazyPopupMenu {
        id: contextMenu

        sourceComponent: PopupMenu {
            id: _contextMenu

            property alias _contentColumn: _contextMenu

            PopupMenuItem {
                visible: root.isPhone
                //% "Call"
                text: qsTrId("components_contacts-me-call")
                icon.source: "image://theme/icon-splus-call"
                onClicked: {
                    if (Telephony.voiceSimUsageMode == Telephony.AlwaysAskSim &&
                            root.simManager.availableModemCount > 0) {
                        simMenu.open()
                    } else {
                        root.startPhoneCall("")
                    }
                }
            }

            PopupMenuItem {
                //% "Send message"
                text: qsTrId("components_contacts-me-send_message")
                opacity: AccessPolicy.smsEnabled ? 1 : Theme.opacityHigh
                icon.source: "image://theme/icon-splus-message"

                onClicked: {
                    if (root.isPhone) {
                        if (AccessPolicy.smsEnabled) {
                            root.startSms()
                        } else {
                            ContactsUtil.showSmsNotice()
                        }
                    } else {
                        root.startInstantMessage()
                    }
                }
            }

            Component.onCompleted: {
                _contextMenu.preferredWidth = AppInternal.UiUtils.getPopupMenuPreferredWidth({
                    popupMenu: _contextMenu,
                    minimumWidth: AppInternal.UiUtils.isLargeScreen
                                    ? _contextMenu.preferredWidth
                                    : root.width,
                    maximumWidth: root.width
                })
            }
        }
    }

    AppInternal.LazyPopupMenu {
        id: simMenu

        sourceComponent: PopupMenu {
            id: _simPopupMenu

            preferredWidth: contextMenu.instance.preferredWidth

            SimPicker {
                id: simPicker

                onSimSelected: {
                    root.startPhoneCall(modemPath)

                    _simPopupMenu.close()
                }
            }
        }
    }

    Item {
        id: rowMain

        height: contentColumn.height
        width: parent.width

        HighlightImage {
            id: icon

            anchors {
                top: parent.top
                topMargin: Theme.paddingLarge
                left: parent.left
                leftMargin: Theme.paddingLarge
            }

            highlighted: root.highlighted
            source: {
                if (root.isMessage) {
                    return "image://theme/icon-s-message"
                } else if (model.eventType === CommCallModel.VoicemailEvent) {
                    return "image://theme/icon-m-voicemail"
                } else {
                    if (model.direction === CommCallModel.Inbound) {
                        return model.isMissedCall
                                ? "image://theme/icon-s-activity-missed-call"
                                : "image://theme/icon-s-activity-incoming-call"
                    } else {
                        return "image://theme/icon-s-activity-outgoing-call"
                    }
                }
            }
        }

        Item {
            id: contentColumn

            anchors {
                top: icon.top
                left: icon.right
                right: timestampColumn.left
                leftMargin: Theme.paddingLarge
                rightMargin: Theme.paddingSmall
            }

            height: titleLabel.height + endpointLabel.height * 2 + Theme.paddingLarge + Theme.paddingMedium + Theme.paddingSmall * 2
            Label {
                id: titleLabel

                anchors {
                    top: parent.top
                    left: parent.left
                    right: parent.right
                }

                text: {
                    if (root.isMessage) {
                        var messageText
                        if (freeText) {
                            messageText = freeText
                        } else if (subject) {
                            messageText = subject
                        } else if (model.eventType == CommHistory.MMSEvent) {
                            //% "Multimedia message"
                            messageText = qsTrId("components_contacts-la-multimedia_message")
                        } else {
                            //% "Message"
                            messageText = qsTrId("components_contacts-la-message")
                        }
                        return messageText
                    } else {
                        return getActionTitle()
                    }
                }

                font.pixelSize: Theme.fontSizeMedium
                color: root.highlighted ? Theme.highlightColor: Theme.primaryColor
                textFormat: Text.PlainText
                truncationMode: Text.ElideRight
                wrapMode: Text.Wrap
                maximumLineCount: 2
            }

            Label {
                id: endpointLabel

                anchors {
                    top: titleLabel.bottom
                    topMargin: Theme.paddingSmall
                    left: parent.left
                    right: parent.right
                }

                text: AsYouTypeFormatterUtil.getFormattedPhoneNumber(model.remoteUid)
                font.pixelSize: Theme.fontSizeExtraSmall
                color: root.highlighted ? Theme.secondaryHighlightColor: Theme.secondaryColor
                truncationMode: TruncationMode.Fade
            }

            ContactActivitySimIndicator {
                id: simIndicator

                anchors {
                    left: durationLabel.visible ? durationLabel.right : titleLabel.left
                    leftMargin: durationLabel.visible ? Theme.paddingSmall : 0
                    verticalCenter: durationLabel.verticalCenter
                }

                simManager: root.simManager
                imsi: model.subscriberIdentity

            }

            HighlightImage {
                id: durationIcon
                visible: !root.isMessage

                anchors {
                    top: endpointLabel.bottom
                    topMargin: Theme.paddingSmall
                    left: titleLabel.left
                    leftMargin: Math.round(-Theme.paddingSmall/2)
                }

                highlighted: root.highlighted
                source: "image://theme/icon-s-duration"
            }

            Label {
                id: durationLabel
                visible: durationIcon.visible

                anchors {
                    verticalCenter: durationIcon.verticalCenter
                    left: durationIcon.right
                    leftMargin: Theme.paddingSmall
                }

                property int duration: (model.endTime.valueOf() - model.startTime.valueOf()) / 1000
                text: Format.formatDuration(duration, duration >= 3600 ? Formatter.DurationLong : Formatter.DurationShort)
                font.pixelSize: Theme.fontSizeExtraSmall
                color: root.highlighted ? Theme.secondaryHighlightColor : Theme.secondaryColor
                truncationMode: TruncationMode.Fade
            }
        }
        Item {
            id: timestampColumn

            anchors {
                top: icon.top
                topMargin: Theme.dp(4)
                right: parent.right
                rightMargin: Theme.paddingLarge
            }
            width: timestampSecondaryLabel.width
            height: timestampLabel.height + timestampSecondaryLabel.height
            Label {
                id: timestampLabel

                anchors {
                    top: parent.top
                    right: parent.right
                }

                text: Format.formatDate(model.endTime, root.formatType)
                font.pixelSize: Theme.fontSizeExtraSmall
                color: root.highlighted ? Theme.secondaryHighlightColor : Theme.secondaryColor
                truncationMode: TruncationMode.Fade
            }

            Label {
                id: timestampSecondaryLabel

                anchors {
                    top: timestampLabel.bottom
                    topMargin: Theme.dp(4)
                    right: parent.right
                }

                text: Format.formatDate(model.endTime, root.showYear ? Format.DateMedium : Format.DateMediumWithoutYear)
                font.pixelSize: Theme.fontSizeExtraSmall
                color: root.highlighted ? Theme.secondaryHighlightColor : Theme.secondaryColor
                truncationMode: TruncationMode.Fade
            }
        }
    }
}
