/**
 * SPDX-FileCopyrightText: 2022-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.6
import Sailfish.Silica 1.0
import Sailfish.Contacts 1.0
import org.nemomobile.contacts 1.0
import Nemo.Configuration 1.0

Dialog {
    id: root

    property string blockedName: ""
    property var contact
    property string phoneNumber
    property var contactIds: []
    property var phoneNumbers: []
    readonly property bool multiSelection: contactIds.length !== 0
    readonly property bool multiPhoneNumbers: phoneNumbers.length !== 0

    property var peopleModel: ContactModelCache.unfilteredModel()

    canAccept: block.canBeAccepted()

    onAccepted: {
        if (multiSelection) {
            var contactsToBlock = []
            for (var i = 0; i < contactIds.length; i++) {
                var selected = peopleModel.personById(contactIds[i])
                if (!!selected) {
                    contactsToBlock.push(selected)
                    selected.blocked = true
                }
            }

            if (!peopleModel.blockPersons(contactsToBlock)) {
                console.error("Cannot block multiple contacts")
                return
            }

            peopleModel.savePeople(contactsToBlock)

            //% "Contacts blocked"
            const notice = qsTrId("components_contacts-block_several_notification_message")
            Notices.show(notice, Notice.Short, Notice.Bottom, 0, -Theme.horizontalPageMargin)
        } else if (multiPhoneNumbers) {
            for (var number in phoneNumbers) {
                if (!ContactModelCache.blockContact(phoneNumbers[number])) {
                    console.error("Cannot block multiple phone numbers")
                    return
                }
            }

            //% "Contacts blocked"
            const notice = qsTrId("components_contacts-block_several_notification_message")
            Notices.show(notice, Notice.Short, Notice.Bottom, 0, -Theme.horizontalPageMargin)
        } else if (block.isBlocked()) {
            if (block.unblockContact()) {
                //% "Contact unblocked"
                const notice = qsTrId("components_contacts-unblock_notification_message")
                Notices.show(notice, Notice.Short, Notice.Bottom, 0, -Theme.horizontalPageMargin)

                if (contact) {
                    contact.blocked = false
                    peopleModel.savePerson(contact)
                }
            }
        } else {
            if (block.blockContact()) {
                //% "Contact blocked"
                const notice = qsTrId("components_contacts-block_notification_message")
                Notices.show(notice, Notice.Short, Notice.Bottom, 0, -Theme.horizontalPageMargin)

                if (contact) {
                    contact.blocked = true
                    peopleModel.savePerson(contact)
                }
            }
        }
    }

    ConfigurationValue {
        id: blockedContacts

        key: "/contactsd/temporaryBlockedContacts"
        defaultValue: ([])
    }

    FontMetrics {
        id: fontMetrics

        font: label.font
    }

    QtObject {
        id: block

        function blockContact() {
            var res = false
            var elapce = Date.now()
            elapce += timePeriod.value * 3600 * 1000
            if (contact && contact.id) {
                res = ContactModelCache.blockContact(contact)

                if (enableTemporaryBlock.checked && res) {
                    var array = blockedContacts.value
                    var contactId = "sql-" + contact.id
                    array.push({"contact": contactId, "period": elapce})
                    blockedContacts.value = array
                }
            } else if (phoneNumber) {
                res = ContactModelCache.blockContact(phoneNumber)

                if (enableTemporaryBlock.checked && res) {
                    var array = blockedContacts.value
                    array.push({"phoneNumber": phoneNumber, "period": elapce})
                    blockedContacts.value = array
                }
            }
            return res
        }

        function unblockContact() {
            var array = blockedContacts.value
            if (contact && contact.id) {
                var key = contact.id;
                for(var id in array) {
                    if (array[id].contact && array[id].contact === key) {
                        array.splice(id, 1)
                        break
                    }
                }
                blockedContacts.value = array
                return ContactModelCache.unblockContact(contact)
            } else if (phoneNumber) {
                for(var id in array) {
                    if (array[id].phoneNumber && array[id].phoneNumber === phoneNumber) {
                        array.splice(id, 1)
                        break
                    }
                }
                blockedContacts.value = array
                return ContactModelCache.unblockContact(phoneNumber)
            } else {
                return false
            }
        }

        function isBlocked() {
            if (contact && contact.id) {
                return ContactModelCache.isBlockedContact(contact)
            } else if (phoneNumber) {
                return ContactModelCache.isBlockedContact(phoneNumber)
            } else if (!multiSelection) {
                console.error("Cannot check contact")
            }
        }

        function getDescription() {
            if (multiSelection) {
                if (isBlocked()) {
                    //% "All calls and messages from these contacts will be unblocked."
                    return qsTrId("components_contacts-unblock_several_message")
                } else {
                    //% "All calls and messages from these contacts will be blocked. You can view them in the settings of the Phone application."
                    return qsTrId("components_contacts-block_several_message")
                }
            } else {
                if (isBlocked()) {
                    //% "All calls and messages from this contact will be unblocked."
                    return qsTrId("components_contacts-unblock_message")
                } else {
                    //% "All calls and messages from this contact will be blocked. You can view them in the settings of the Phone application."
                    return qsTrId("components_contacts-block_message")
                }
            }
        }

        function cutName(name) {
            return fontMetrics.elidedText(name, Text.ElideRight, (root.width - 2 * Theme.horizontalPageMargin - Theme.paddingLarge) * 2)
        }

        function getBlockedName() {
            var message = ""
            if (multiSelection) {
                //% "Are you sure you want to block contacts (%n)?"
                message = qsTrId("components_contacts-block_several_title", contactIds.length)
            } else if (multiPhoneNumbers) {
                //% "Are you sure you want to block contacts (%n)?"
                message = qsTrId("components_contacts-block_several_title", phoneNumbers.length)
            } else if (!block.isBlocked()) {
                //% "Do you want to block %1?"
                message = qsTrId("components_contacts-block_message_title").arg(cutName(blockedName))
            } else {
                //% "Do you want to unblock %1?"
                message = qsTrId("components_contacts-unblock_message_title").arg(cutName(blockedName))
            }

            return message
        }

        function canBeAccepted() {
            if (phoneNumber) {
                return !PeopleModelUtils.isShortNumber(phoneNumber)
            } else if (contact && contact.id) {
                var phones = contact.phoneDetails
                for (var i in phones) {
                    var _phoneNumber = phones[i].number
                    if (PeopleModelUtils.isShortNumber(_phoneNumber)) {
                        return false
                    }
                }
                return true
            } else if (multiSelection) {
                var result = false
                for (var j in contactIds) {
                    var person = root.peopleModel.personById(contactIds[j])
                    var _phones = person.phoneDetails
                    for (var jj in _phones) {
                        var __phoneNumber = _phones[jj].number
                        if (PeopleModelUtils.isShortNumber(__phoneNumber)) {
                            return false
                        }
                        result = true
                    }
                }
                return result
            } else if (multiPhoneNumbers) {
                for (var phone in phoneNumbers) {
                    if (PeopleModelUtils.isShortNumber(phoneNumbers[phone])) {
                        return false
                    }
                }
                return true
            }
            return false
        }
    }

    DialogHeader {
        id: header

        acceptText: multiPhoneNumbers || multiSelection || !block.isBlocked()
                    //% "Block"
                    ? qsTrId("components_contacts-block_accept_text")
                    //% "Unblock"
                    : qsTrId("components_contacts-unblock_accept_text")
    }

    Text {
        id: label

        anchors {
            top: header.bottom
            left: parent.left
            leftMargin: Theme.horizontalPageMargin
            right: parent.right
            rightMargin: Theme.horizontalPageMargin
        }

        text: block.getBlockedName()
        clip: true
        wrapMode: Text.Wrap
        font.pixelSize: Theme.fontSizeExtraLarge
        color: Theme.primaryColor
        maximumLineCount: 5
    }

    Text {
        id: name

        anchors {
            top: label.bottom
            topMargin: Theme.paddingLarge
            left: parent.left
            leftMargin: Theme.horizontalPageMargin
            right: parent.right
            rightMargin: Theme.horizontalPageMargin
        }

        text: block.getDescription()
        wrapMode: Text.WordWrap
        font.pixelSize: Theme.fontSizeMedium
        color: Theme.secondaryColor
    }

    Column {
        id: temporaryBlock

        anchors {
            top: name.bottom
            topMargin: Theme.paddingLarge
            left: parent.left
            leftMargin: Theme.horizontalPageMargin
            right: parent.right
            rightMargin: Theme.horizontalPageMargin
        }

        TextSwitch {
            id: enableTemporaryBlock

            //% "Temporary block"
            text: qsTrId("components_contacts-temporary_block_switch")
            visible: !multiPhoneNumbers && !multiSelection && !block.isBlocked()
        }

        Slider {
            id: timePeriod

            width: parent.width
            enabled: enableTemporaryBlock.checked
            opacity: enableTemporaryBlock.checked ? 1.0 : Theme.opacityFaint
            visible: enableTemporaryBlock.visible
            //% "The contact will be blocked"
            label: qsTrId("components_contacts-temporary_block_slider_label")
            //% "%n hour(s)"
            valueText: qsTrId("components_contacts-temporary_block_slider_value", value)

            //All values below measured in hours
            stepSize: 1
            minimumValue: 1
            maximumValue: 48
            value: 3
        }
    }
}
