/*
 * SPDX-FileCopyrightText: 2019-2021 Jolla Ltd
 * SPDX-FileCopyrightText: 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.0
import Sailfish.Silica 1.0
import Sailfish.Bluetooth 1.0
import org.kde.bluezqt 1.0 as BluezQt

ColumnView {
    id: columnView

    property var filters
    property var excludedDevices: []
    property bool highlightSelectedDevice
    property bool openMenuOnPressAndHold

    property var _connectingDevices: []
    property string _selectedDevice: ""

    signal deviceItemClicked(string address, bool paired)
    signal deviceSettingsClicked(string address)
    signal removeDeviceClicked(string address)

    function addConnectingDevice(addr) {
        addr = addr.toUpperCase()
        for (var i=0; i<_connectingDevices.length; i++) {
            if (_connectingDevices[i].toUpperCase() == addr) {
                return
            }
        }
        var devices = _connectingDevices
        devices.push(addr)
        _connectingDevices = devices
    }

    function removeConnectingDevice(addr) {
        addr = addr.toUpperCase()
        var devices = _connectingDevices
        for (var i=0; i<devices.length; i++) {
            if (devices[i].toUpperCase() == addr) {
                devices.splice(i, 1)
                _connectingDevices = devices
                return
            }
        }
    }

    function clearSelectedDevice() {
        _selectedDevice = ""
    }

    function _matchesProfileHint(profiles, classOfDevice) {
        return preferredProfileHint < 0
                || BluetoothProfiles.profileMatchesDeviceProperties(preferredProfileHint, profiles, classOfDevice)
    }

    itemHeight: Theme.itemSizeSmall
    model: BluezQt.DevicesModel {
        id: devicesModel
        filters: columnView.filters
        hiddenAddresses: columnView.excludedDevices
    }

    delegate: ListItem {
        id: deviceDelegate

        property bool showConnectionStatus: model.Connected || isConnecting || minConnectionStatusTimeout.running
        property bool isConnecting: _connectingDevices.indexOf(model.Address.toUpperCase()) >= 0

        property bool matchesProfileHint: columnView._matchesProfileHint(model.Uuids, model.Class)
        property bool useHighlight: highlighted
                                    || (highlightSelectedDevice && model.Address === columnView._selectedDevice)

        readonly property bool isBatteryVisible: !!model.Battery
        readonly property int lowBatteryPercentage: 10
        readonly property int midBatteryPercentage: 40
        readonly property int highBatteryPercentage: 70

        acceptedButtons: Qt.LeftButton | Qt.RightButton

        openMenuOnPressAndHold: columnView.openMenuOnPressAndHold

        onIsConnectingChanged: {
            if (isConnecting) {
                minConnectionStatusTimeout.start()
            }
        }

        onClicked: {
            if (mouse.button === Qt.RightButton) {
                openMenu()
                return
            }

            columnView._selectedDevice = model.Address
            columnView.deviceItemClicked(model.Address, model.Paired)
        }

        Timer {
            id: minConnectionStatusTimeout
            interval: 2000
        }

        menu: Component {
            ContextMenu {
                MenuItem {
                    text: {
                        if (model.Paired) {
                            //: Show settings for the selected bluetooth device
                            //% "Device settings"
                            return qsTrId("components_bluetooth-me-device_settings")
                        } else {
                            //: Show info for the selected bluetooth device
                            //% "Details"
                            return qsTrId("components_bluetooth-me-details")
                        }

                    }

                    onClicked: columnView.deviceSettingsClicked(model.Address)
                }

                MenuItem {
                    text: {
                        if (model.Paired) {
                            //: Remove the pairing with the selected bluetooth device
                            //% "Remove pairing"
                            return qsTrId("components_bluetooth-me-pairing_remove")
                        } else {
                            //: Forget the selected bluetooth device
                            //% "Forget device"
                            return qsTrId("components_bluetooth-me-device_forget")
                        }
                    }

                    onClicked: {
                        if (model.Address === columnView._selectedDevice) {
                            columnView._selectedDevice = ""
                        }
                        columnView.removeDeviceClicked(model.Address)
                    }
                }
            }
        }

        BluetoothDeviceInfo {
            id: deviceInfo
            address: model.Address
            deviceClass: model.Class
        }

        Image {
            id: icon
            x: Theme.horizontalPageMargin
            anchors.verticalCenter: parent.verticalCenter
            source: "image://theme/" + deviceInfo.icon + (deviceDelegate.useHighlight ? "?" + Theme.highlightColor : "")
            opacity: deviceDelegate.matchesProfileHint || deviceDelegate.useHighlight ? 1 : Theme.opacityHigh
        }

        Label {
            id: deviceNameLabel

            anchors {
                left: icon.right
                leftMargin: Theme.paddingMedium
                right: pairedCheckMark.left
                rightMargin: Theme.paddingMedium
            }

            y: deviceDelegate.contentHeight/2 - implicitHeight/2
               - (showConnectionStatus ? connectedLabel.implicitHeight/2 : 0)
            text: model.FriendlyName
            truncationMode: TruncationMode.Fade
            color: deviceDelegate.useHighlight
                   ? Theme.highlightColor
                   : Theme.rgba(Theme.primaryColor, deviceDelegate.matchesProfileHint ? 1.0 : Theme.opacityHigh)

            Behavior on y { NumberAnimation {} }
        }

        Label {
            id: connectedLabel

            anchors {
                left: deviceNameLabel.left
                top: deviceNameLabel.bottom
            }

            opacity: showConnectionStatus ? 1.0 : 0.0
            color: deviceDelegate.useHighlight
                   ? Theme.secondaryHighlightColor
                   : Theme.secondaryColor
            text: model.Connected
                    //% "Connected"
                  ? qsTrId("components_bluetooth-la-connected")
                  : deviceDelegate.isConnecting || minConnectionStatusTimeout.running
                      //% "Connecting"
                    ? qsTrId("components_bluetooth-la-connecting")
                    : ""
            font.pixelSize: Theme.fontSizeExtraSmall

            Behavior on opacity { FadeAnimation {} }
        }

        Label {
            id: commaLabel

            anchors {
                left: connectedLabel.right
                top: deviceNameLabel.bottom
            }

            opacity: isBatteryVisible ? 1.0 : 0.0
            color: deviceDelegate.useHighlight
                   ? Theme.secondaryHighlightColor
                   : Theme.secondaryColor
            text: ","
            font.pixelSize: Theme.fontSizeExtraSmall
        }

        Icon {
            id: batteryPercentageIcon

            function percentageIconName() {
                const batteryPercentage = model.Battery.percentage
                if (!batteryPercentage) {
                    return ""
                }

                if (batteryPercentage <= deviceDelegate.lowBatteryPercentage) {
                    return "icon-s-battery-0"
                }

                if (batteryPercentage <= deviceDelegate.midBatteryPercentage) {
                    return "icon-s-battery-10"
                }

                if (batteryPercentage <= deviceDelegate.highBatteryPercentage) {
                    return "icon-s-battery-40"
                }

                return "icon-s-battery"
            }

            anchors {
                left: commaLabel.right
                leftMargin: Theme.paddingSmall
                top: deviceNameLabel.bottom
                verticalCenter: connectedLabel.verticalCenter
            }

            opacity: isBatteryVisible ? 1.0 : 0.0
            color: Qt.lighter(Theme.secondaryColor, Theme.opacityHigh)
            highlightColor: Qt.lighter(Theme.secondaryHighlightColor, Theme.opacityHigh)
            source: isBatteryVisible ? "image://theme/" + percentageIconName() : ""
        }

        Label {
            id: batteryPercentageLabel

            anchors {
                leftMargin: Theme.dp(2)
                left: batteryPercentageIcon.right
                top: deviceNameLabel.bottom
            }

            opacity: isBatteryVisible ? 1.0 : 0.0
            color: deviceDelegate.useHighlight
                   ? Theme.secondaryHighlightColor
                   : Theme.secondaryColor

            text: (model.Connected && isBatteryVisible) ? qsTr("%1 %").arg(model.Battery.percentage) : ""
            font.pixelSize: Theme.fontSizeExtraSmall

            Behavior on opacity { FadeAnimation {} }
        }

        Image {
            id: pairedCheckMark

            anchors {
                right: parent.right
                rightMargin: Theme.horizontalPageMargin
                verticalCenter: parent.verticalCenter
            }

            visible: model.Paired && model.Trusted
            source: "image://theme/icon-s-accept" + (deviceDelegate.useHighlight ? "?" + Theme.highlightColor : "")
            opacity: icon.opacity
        }
    }
}
