/****************************************************************************************
**
** SPDX-FileCopyrightText: 2024 - 2025 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: Proprietary
**
****************************************************************************************/

import QtQuick 2.6
import QtQuick.Window 2.2 as QtQuick
import Sailfish.Silica 1.0
import Aurora.Controls.private 1.0
import QtGraphicalEffects 1.0

PullToRefreshView {
    id: root

    function _getViewGestureProgress() {
        return internal.gestureProgress
    }

    function _showRefreshStatus(success) {
        statusDisplayTimer.restart()
        internal.statusText = success
                //% "Refreshed"
                ? qsTrId("pull_to_refresh-me-refreshed")
                //% "Refresh failed"
                : qsTrId("pull_to_refresh-me-refresh_failed")
        internal.statusIconSource = success
                ? "image://theme/icon-s-accept"
                : "image://theme/icon-s-error"
    }

    function _showRefreshCustomStatus(iconSource, message) {
        if (iconSource != "" || message != "") {
            statusDisplayTimer.restart()
        }

        internal.statusText = message
        internal.statusIconSource = iconSource
    }

    function _showDescription() {
        internal.statusText = Qt.binding(function () {
            return viewData.description !== ""
                    ? viewData.description
                    //% "Pull to refresh"
                    : qsTrId("pull_to_refresh-me-pull_to_refresh_description")
        })
        internal.statusIconSource = ""
    }

    Connections {
        onGestureActiveChanged: {
            if (viewData.gestureActive) {
                _showDescription()
            }
        }

        target: viewData
    }

    QtObject {
        id: internal

        readonly property bool atTriggerPoint: gestureProgress >= 1.0
        readonly property bool noStatus: statusText == "" && statusIconSource == ""
        property real gestureProgress
        property QtObject ngfEffectHighlight
        property string statusText
        property url statusIconSource

        onAtTriggerPointChanged: {
            if (atTriggerPoint && viewData.pressed && internal.ngfEffectHighlight) {
                internal.ngfEffectHighlight.play()
            }
        }

        Behavior on gestureProgress {
            id: gestureProgressBehavior

            SequentialAnimation {
                NumberAnimation { duration: 200 }
            }
        }
    }

    Loader {
        anchors.fill: parent

        asynchronous: true
        sourceComponent: uiComponent
    }

    Timer {
        id: statusDisplayTimer

        interval: 3000
    }

    Component {
        id: defaultBusyIndicatorComponent

        BusyIndicator {
            // TODO: refactoring within the scope of OS-32712
            opacity: 1
            size: BusyIndicatorSize.Small
        }
    }

    Component {
        id: uiComponent

        Item {
            id: ui

            opacity: internal.gestureProgress * 10.0
            visible: opacity > 0.0

            RectangularGlow {
                anchors.fill: statusUiBackground
                anchors.margins: -Theme.dp(1)

                cornerRadius: Theme.dp(24)
                glowRadius: cornerRadius
                color: Theme.colorScheme === Theme.DarkOnLight ? "#19000000" : "#33000000"
            }

            Rectangle {
                id: statusUiBackground

                anchors.centerIn: statusUi

                width: statusUi.width
                height: statusUi.height
                color: Theme._highlightBackgroundAlternativeColor
                radius: height * 0.5

                Behavior on width {
                    enabled: internal.atTriggerPoint || statusDisplayTimer.running

                    SequentialAnimation {
                        PropertyAction { target: statusUi; property: "opacity"; value: 0.0 }
                        NumberAnimation { duration: 300; easing.type: Easing.InOutCubic }
                        PropertyAction { target: statusUi; property: "opacity"; value: 1.0 }
                    }
                }
            }

            Row {
                id: statusUi

                anchors.horizontalCenter: parent.horizontalCenter

                y: Theme.paddingMedium + internal.gestureProgress * 0.25 * QtQuick.Screen.pixelDensity * 30
                height: Theme.dp(62)
                spacing: Theme.paddingMedium
                rightPadding: viewData.refreshing || internal.statusText == ""
                              ? Theme.dp(15)
                              : Theme.dp(26)
                leftPadding: rightPadding

                Item {
                    anchors.verticalCenter: parent.verticalCenter

                    width: Theme.iconSizeSmall
                    height: Theme.iconSizeSmall
                    visible: viewData.refreshing || internal.noStatus || icon.status !== Image.Null

                    HighlightImage {
                        id: icon

                        anchors.fill: parent

                        visible: !viewData.refreshing && status !== Image.Null
                        source: internal.statusIconSource
                    }

                    Loader {
                        anchors.fill: parent

                        visible: viewData.refreshing || internal.noStatus
                        sourceComponent: root.refreshIndicator
                                         ? root.refreshIndicator
                                         : defaultBusyIndicatorComponent
                    }
                }

                Label {
                    anchors.verticalCenter: parent.verticalCenter

                    width: Math.min(ui.width - Theme.paddingLarge * 2 - Theme.dp(26) * 2 - x, implicitWidth)
                    textFormat: Text.PlainText
                    truncationMode: TruncationMode.Fade
                    font.pixelSize: Theme.fontSizeExtraSmall
                    horizontalAlignment: Text.AlignLeft
                    text: internal.statusText
                    visible: !viewData.refreshing
                }
            }

            RectangularGlow {
                anchors.fill: handle
                anchors.margins: -Theme.dp(1)

                opacity: handle.opacity
                cornerRadius: Theme.dp(24)
                glowRadius: cornerRadius
                color: Theme.colorScheme === Theme.DarkOnLight ? "#19000000" : "#33000000"
            }

            Rectangle {
                id: handle

                anchors.horizontalCenter: parent.horizontalCenter

                y: Theme.paddingMedium + statusUi.height + internal.gestureProgress * QtQuick.Screen.pixelDensity * 30
                width: Theme.dp(62)
                height: Theme.dp(42)
                radius: height * 0.5
                color: Theme._highlightBackgroundAlternativeColor
                opacity: viewData.gestureActive ? 1.0 : 0.0

                HighlightImage {
                    anchors.centerIn: parent

                    source: "image://theme/icon-s-down-alt"
                }

                Connections {
                    onGestureActiveChanged: {
                        if (viewData.gestureActive) {
                            handle.opacity = 1.0
                        }
                    }
                    onRefreshingChanged: {
                        if (viewData.refreshing) {
                            handle.opacity = 0.0
                        }
                    }

                    target: viewData
                }

                Behavior on opacity {
                    enabled: internal.atTriggerPoint

                    FadeAnimation { }
                }
            }
        }
    }

    states: [
        State {
            name: "dragging"
            when: viewData.pressed && viewData.gestureActive
            StateChangeScript {
                script: {
                    gestureProgressBehavior.enabled = false
                    internal.gestureProgress = Qt.binding(function () { return root.gestureProgress })
                }
            }
        }, State {
            name: "captured"
            when: viewData.pressed && !viewData.refreshing && !statusDisplayTimer.running
            StateChangeScript {
                script: {
                    gestureProgressBehavior.enabled = true
                    internal.gestureProgress = internal.gestureProgress
                }
            }
        }, State {
            name: "refreshing"
            when: viewData.refreshing || statusDisplayTimer.running
            StateChangeScript {
                script: {
                    gestureProgressBehavior.enabled = true
                    internal.gestureProgress = 0.1
                }
            }
        }, State {
            name: ""
            StateChangeScript {
                script: {
                    gestureProgressBehavior.enabled = true
                    internal.gestureProgress = 0.0
                }
            }
        }
    ]

    Component.onCompleted: {
        internal.ngfEffectHighlight
                = Qt.createQmlObject("import org.nemomobile.ngf 1.0; NonGraphicalFeedback { event: 'pulldown_highlight' }",
                                     root, 'NonGraphicalFeedback')
    }
}
