/*
 * SPDX-FileCopyrightText: 2024-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.0
import Sailfish.Silica 1.0
import Sailfish.Lipstick 1.0
import Aurora.AppActivity 1.0

Item {
    id: root

    property variant currentIndicators: ([])

    function iconPathActivity(type) {
        switch (type) {
            case AppActivityManager.activityCamera:
                return "./icons/icon-activity-camera.svg"
            case AppActivityManager.activityMicrophone:
                return "./icons/icon-activity-microphone.svg"
            case AppActivityManager.activityGeolocation:
                return "./icons/icon-activity-location.svg"
            case AppActivityManager.activityScreen:
                return "./icons/icon-activity-screen.svg"
            default:
                return ""
        }
    }

    function addIndicatorView(type) {
        var newIndicator
        for (var i = 0; i < indicators.count; i++)  {
            newIndicator = indicators.itemAt(i)
            if (newIndicator.state == "hidden") {
                break
            }
        }

        currentIndicators.push(newIndicator)
        var margins = 0
        for (var i = currentIndicators.length - 1; i >= 0; i--)  {
            currentIndicators[i].anchors.leftMargin = margins
            margins += (Screen._isTablet ? Theme.dp(8) : Theme.paddingSmall) + currentIndicators[i].sourceSize.width
        }

        newIndicator.state = "visible"
        newIndicator.type = type
    }

    function removeIndicatorView(removeIndex) {
        var removeIndicator = currentIndicators[removeIndex]
        var margins = removeIndicator.anchors.leftMargin
        removeIndicator.state = "hidden"
        for (var i = removeIndex - 1; i >= 0; i--)  {
            currentIndicators[i].anchors.leftMargin = margins
            margins += (Screen._isTablet ? Theme.dp(8) : Theme.paddingSmall) + currentIndicators[i].sourceSize.width
        }

        currentIndicators.splice(removeIndex, 1)
    }

    function refreshBackgroundItemWidth() {
        var textOffset = textContainer.width > 0 ? Theme.paddingMedium : 0
        backgroundItem.width = (Screen._isTablet ? Theme.paddingMedium : Theme.dp(8))
                            + indicatorsContainer.width
                            + Theme.paddingMedium
                            + textContainer.width
                            + textOffset
    }

    function refreshIconsItemWidth() {
        if (root.state === "visible") {
            indicatorsContainer.width = ((Screen._isTablet ? Theme.iconSizeSmallPlus : Theme.iconSizeSmall) * currentIndicators.length)
                                    + ((Screen._isTablet ? Theme.dp(8) : Theme.paddingSmall) * (currentIndicators.length - 1))
        } else if (root.state === "hidden") {
            indicatorsContainer.width = Screen._isTablet ? Theme.iconSizeSmallPlus : Theme.iconSizeSmall
        }
    }

    function enableIndicators() {
        currentIndicators = []

        for (var i = 0; i < AppActivityManager.activeNotificationsTypes.length; i++) {
            addIndicatorView(AppActivityManager.activeNotificationsTypes[i])
        }

        refreshIconsItemWidth()
    }

    function updateIndicators() {
        for (var i = 0; i < currentIndicators.length; i++)  {
            var needRemove = true
            for (var j = 0; j < AppActivityManager.activeNotificationsTypes.length; j++) {
                if (currentIndicators[i].type == AppActivityManager.activeNotificationsTypes[j]) {
                    needRemove = false
                    break
                }
            }

            if (needRemove) {
                removeIndicatorView(i)
            }
        }

        for (var i = 0; i < AppActivityManager.activeNotificationsTypes.length; i++) {
            var needAdd = true

            for (var j = 0; j < currentIndicators.length; j++) {
                if (currentIndicators[j].type == AppActivityManager.activeNotificationsTypes[i]) {
                    needAdd = false
                    break
                }
            }

            if (needAdd) {
                addIndicatorView(AppActivityManager.activeNotificationsTypes[i])
            }
        }

        refreshIconsItemWidth()
    }

    function updateBubble() {
        if (textApplications.state != "scroll") {
            updateIndicators()
            updateText()
        }
    }

    function textContainerWidth() {
        var maxWidth = root.width
                    - Theme.paddingLarge * 2
                    - indicatorsContainer.width
                    - Theme.paddingMedium * 2
                    - (Screen._isTablet ? Theme.paddingMedium : Theme.dp(8))

        return Math.min(maxWidth, textApplications.implicitWidth)
    }

    function updateText() {
        textApplications.text = AppActivityManager.applicationsString()
        if (textContainer.state === "hidden") {
            textContainer.state = "visible"
        } else if (textContainer.state === "visible") {
            textContainer.state = "change"
        }

        textContainer.width = 0
        textContainer.width = textContainerWidth()
        if (textApplications.implicitWidth > textContainerWidth()) {
            scrollTextTimer.start()
            textApplications.truncate = true
        } else {
            textApplications.truncate = false
        }
    }

    function heightForGridCell(index) {
        if (index < 0 || index >= AppActivityManager.applications.rowCount()) {
            return 0
        }

        var applications = AppActivityManager.applications
        var indicatorsCount = applications.data(applications.index(index, 0), Qt.UserRole + 1).length
        var maxIndicatorsCount = 4

        return Theme.dp(Screen._isTablet ? 426 : 318) - Theme.dp(Screen._isTablet ? 60 : 44) * (maxIndicatorsCount - indicatorsCount)
    }

    width: parent.width
    height: backgroundItem.height
    state: "hidden"
    onWidthChanged: {
        textApplications.state = ""
        updateBubble()
    }

    BackgroundItem {
        id: backgroundItem

        anchors.horizontalCenter: parent.horizontalCenter
        width: (Screen._isTablet ? Theme.paddingMedium : Theme.dp(8)) + indicatorsContainer.width + Theme.paddingMedium
        height: 0
        opacity: Theme.opacityFaint
        _backgroundRadius: height * 0.5
        highlightedColor: Theme.highlightColor

        onClicked: activityPopup.showPopup()

        Rectangle {
            anchors.fill:parent
            color: Theme.primaryColor
            radius: backgroundItem._backgroundRadius
        }

        Behavior on height {
            PropertyAnimation {
                duration: 200
                easing.type: Easing.InOutQuad
            }
        }

        Behavior on width {
            PropertyAnimation {
                duration: 200
                easing.type: Easing.InOutQuad
            }
        }
    }

    Item {
        id: indicatorsContainer

        anchors {
            verticalCenter: backgroundItem.verticalCenter
            left: backgroundItem.left
            leftMargin: Screen._isTablet ? Theme.paddingMedium : Theme.dp(8)
        }

        width: Screen._isTablet ? Theme.iconSizeSmallPlus : Theme.iconSizeSmall

        Repeater {
            id: indicators

            model: 4
            delegate: Image {
                id: delegate

                property string type

                anchors {
                    verticalCenter: parent.verticalCenter
                    left: parent.left
                }

                width: Screen._isTablet ? Theme.iconSizeSmallPlus : Theme.iconSizeSmall
                height: width
                sourceSize.width: width
                sourceSize.height: width
                source: iconPathActivity(type)
                state: "hidden"

                states: [
                    State {
                        name: "hidden"
                        PropertyChanges {
                            target: delegate
                            scale: 0
                            opacity: 0
                        }
                    },
                    State {
                        name: "visible"
                        PropertyChanges {
                            target: delegate
                            scale: 1
                            opacity: 1
                        }
                    }
                ]

                transitions: Transition {
                    ParallelAnimation {
                        PropertyAnimation {
                            property: "scale"
                            duration: 200
                            easing.type: Easing.InOutQuad
                        }
                        PropertyAnimation {
                            property: "opacity"
                            duration: 200
                            easing.type: Easing.Linear
                        }
                    }
                    onRunningChanged: {
                        if (!running && delegate.state == "hidden") {
                            delegate.anchors.leftMargin = 0
                        }
                    }
                }

                Behavior on anchors.leftMargin {
                    PropertyAnimation {
                        duration: 200
                        easing.type: Easing.InOutQuad
                    }
                }
            }
        }
    }

    Timer {
        id: scrollTextTimer

        interval: 2800

        onTriggered: {
            if (root.state === "visible"
                && textApplications.implicitWidth > textContainer.width) {

                textApplications.state = ""
                textApplications.scrollShift = textApplications.implicitWidth
                textApplications.edgeFade = textApplications.implicitWidth
                var text = textApplications.text
                var spacingText = "          "
                textApplications.text = spacingText
                textApplications.scrollShift += textApplications.implicitWidth
                textApplications.text = text + spacingText + text
                textApplications.state = "scroll"
            }
        }
    }

    Item {
        id: textContainer

        anchors {
            left: backgroundItem.left
            leftMargin: indicatorsContainer.width + (Screen._isTablet ? Theme.paddingMedium : Theme.dp(8)) + Theme.paddingMedium
            verticalCenter: backgroundItem.verticalCenter
        }

        height: parent.height
        clip: true
        state: "hidden"

        Label {
            id: textApplications

            property int scrollShift
            property int edgeFade
            property bool truncate

            anchors.verticalCenter: parent.verticalCenter
            width: parent.width
            font.pixelSize: Screen._isTablet ? Theme.fontSizeSmall : Theme.fontSizeExtraSmall
            textFormat: Text.PlainText
            maximumLineCount: 1
            color: backgroundItem.highlighted ? Theme.highlightColor : Theme.primaryColor

            states: [
                State {
                    name: "scroll"
                    PropertyChanges {
                        target: textApplications
                        x: -scrollShift
                    }
                },
                State {
                    name: ""
                    PropertyChanges {
                        target: textApplications
                        x: 0
                    }
                }
            ]

            transitions: Transition {
                to: "scroll"
                SmoothedAnimation {
                    property: "x"
                    easing.type: Easing.InOutQuad
                }
                onRunningChanged: {
                    if (!running) {
                        textApplications.state = ""
                        updateBubble()
                    }
                }
            }
        }

        layer.enabled: textApplications.truncate
        layer.effect: OpacityRampEffectBase {
            id: fadeEffect

            anchors.fill: textContainer
            offset: 0.95
            slope: 25
            direction: OpacityRamp.LeftToRight
            layer.enabled: textApplications.x < -5 && textApplications.x > -textApplications.edgeFade
            layer.effect: OpacityRampEffect {
                direction: OpacityRamp.RightToLeft
                offset: fadeEffect.offset
                slope: fadeEffect.slope
                sourceItem: fadeEffect
            }
        }

        states: [
            State {
                name: "hidden"
                PropertyChanges {
                    target: textContainer
                    width: 0
                }
            }
        ]

        Behavior on width {
            SequentialAnimation {
                PauseAnimation { duration: textContainer.state === "visible" ? 250 : 0 }
                ScriptAction {
                    script: {
                        if (textContainer.state === "hidden") {
                            scrollTextTimer.stop()
                            textApplications.state = ""
                            backgroundItem.width = (Screen._isTablet ? Theme.paddingMedium : Theme.dp(8))
                                                + indicatorsContainer.width
                                                + Theme.paddingMedium
                        } else {
                            backgroundItem.width = (Screen._isTablet ? Theme.paddingMedium : Theme.dp(8))
                                                + indicatorsContainer.width
                                                + Theme.paddingMedium
                                                + textContainerWidth()
                                                + Theme.paddingMedium
                        }
                    }
                }
                PauseAnimation { duration: textContainer.state === "visible" ? 20 : 0 }
                PropertyAnimation {
                    duration: 200
                    easing.type: Easing.InOutQuad
                }
                ScriptAction {
                    script: {
                        if (textContainer.state === "hidden") {
                            textApplications.text = ""
                            textApplications.x = 0
                            updateIndicators()
                            backgroundItem.height = 0
                            root.state = "hidden"
                        }
                    }
                }
            }
        }

        Behavior on anchors.leftMargin {
            PropertyAnimation {
                duration: 200
                easing.type: Easing.InOutQuad
            }
        }
    }

    PopupPageItem {
        id: activityPopup

        property int componentMaxHeight: Screen.height * 0.66 - Theme.paddingLarge * 2 - Theme.fontSizeLarge

        //: The title of the application activity pop-up window
        //% "Application activity"
        label: qsTrId("app-activity-title")

        showFromCenter: true
        contentMargins: Theme.paddingMedium
        labelMargins: Theme.paddingLarge

        contentComponent: Component {
            SilicaFlickable {
                height: grid.height
                contentHeight: grid.height
                clip: true
                onHeightChanged: {
                    activityPopup.stretchContentToPopup = height > activityPopup.componentMaxHeight
                }

                Grid {
                    id: grid

                    width: parent.width
                    columns: 2
                    spacing: Theme.paddingMedium

                    Repeater {
                        model: AppActivityManager.applications
                        delegate: Item {
                            width: {
                                var value = grid.width
                                if (AppActivityManager.applications.rowCount() > 1) {
                                    value = (value - Theme.paddingMedium) * 0.5
                                }

                                return value;
                            }
                            height: Math.max(heightForGridCell(index), heightForGridCell(index % 2 === 0 ? index + 1 : index - 1))

                            Rectangle {
                                anchors.fill: parent
                                color: Theme.overlayBackgroundColor
                                radius: Theme.paddingMedium
                                opacity: 0.04
                                border.color: Theme.primaryColor
                                border.width: Theme.dp(2)
                            }

                            Column {
                                anchors {
                                    fill: parent
                                    leftMargin: Theme.paddingLarge
                                    rightMargin: Theme.paddingLarge
                                    bottomMargin: Theme.paddingLarge
                                }

                                Row {
                                    width: parent.width
                                    height: processIcon.height + (Theme.paddingLarge * 2)
                                    spacing: Theme.paddingMedium

                                    LauncherIcon {
                                        id: processIcon

                                        anchors.verticalCenter: parent.verticalCenter
                                        size: Screen._isTablet ? Theme.iconSizeLauncher : Theme.iconSizeSmallPlus
                                        icon: iconPath
                                        scaleSource: false
                                    }

                                    Label {
                                        anchors.verticalCenter: parent.verticalCenter
                                        width: {
                                            var calculateWidth = parent.width
                                            calculateWidth -= Theme.iconSizeSmallPlus
                                            calculateWidth -= Theme.paddingMedium
                                            return Math.min(implicitWidth, calculateWidth)
                                        }
                                        text: name
                                        color: Theme.primaryColor
                                        font.pixelSize: Screen._isTablet ? Theme.fontSizeLarge : Theme.fontSizeExtraSmall
                                        textFormat: Text.PlainText
                                        maximumLineCount: 1
                                        truncationMode: TruncationMode.Fade
                                    }
                                }

                                Label {
                                    width: Math.min(implicitWidth, parent.width)
                                    opacity: Theme.opacityHigh
                                    //: The title of the activity used
                                    //% "Used:"
                                    text: qsTrId("app-activity-used")
                                    color: Theme.primaryColor
                                    font.pixelSize: Screen._isTablet ? Theme.fontSizeExtraSmall : Theme.fontSizeTiny
                                    textFormat: Text.PlainText
                                    maximumLineCount: 1
                                }

                                Item {
                                    width: parent.width
                                    height: Theme.paddingMedium
                                }

                                Repeater {
                                    model: currentActivity
                                    delegate: Column {
                                        Row {
                                            anchors.topMargin: Theme.paddingLarge
                                            spacing: Theme.paddingMedium

                                            Image {
                                                anchors.verticalCenter: parent.verticalCenter
                                                width: Screen._isTablet ? Theme.iconSizeSmallPlus : Theme.iconSizeSmall
                                                height: width
                                                sourceSize.width: width
                                                sourceSize.height: width
                                                source: iconPathActivity(modelData)
                                            }

                                            Label {
                                                anchors.verticalCenter: parent.verticalCenter
                                                width: Math.min(implicitWidth, parent.width - Theme.iconSizeSmall)
                                                text: {
                                                    switch (modelData) {
                                                        case AppActivityManager.activityCamera:
                                                            //: The name of the camera activity
                                                            //% "Camera"
                                                            return qsTrId("app-activity-camera")
                                                        case AppActivityManager.activityMicrophone:
                                                            //: The name of the microphone activity
                                                            //% "Microphone"
                                                            return qsTrId("app-activity-microphone")
                                                        case AppActivityManager.activityGeolocation:
                                                            //: The name of the geolocation activity
                                                            //% "Geolocation"
                                                            return qsTrId("app-activity-geolocation")
                                                        case AppActivityManager.activityScreen:
                                                            //: The name of the screen activity
                                                            //% "Screen"
                                                            return qsTrId("app-activity-screen")
                                                        default:
                                                            //: The name of the undefined activity
                                                            //% "undefined"
                                                            return qsTrId("app-activity-undefined")
                                                    }
                                                }
                                                color: Theme.primaryColor
                                                font.pixelSize: Screen._isTablet ? Theme.fontSizeMedium : Theme.fontSizeExtraSmall
                                                textFormat: Text.PlainText
                                                maximumLineCount: 1
                                            }
                                        }

                                        Item {
                                            width: parent.width
                                            height: Theme.paddingMedium
                                            visible: index != currentActivity.length
                                        }
                                    }
                                }
                            }
                        }
                    }
                }

                VerticalScrollDecorator {}

                Behavior on height {
                    NumberAnimation {
                        duration: 200
                        easing.type: Easing.InOutQuad
                    }
                }
            }
        }
    }

    Connections {
        target: AppActivityManager

        onApplicationsChanged: {
            if (AppActivityManager.applications.rowCount() == 0) {
                activityPopup.hidePopup()
            }
        }

        onActiveNotificationsTypesChanged: {
            if (AppActivityManager.activeNotificationsTypes.length === 0) {
                textContainer.state = "hidden"
            } else if (root.state === "hidden") {
                root.state = "visible"
                enableIndicators()
                refreshBackgroundItemWidth()
                updateText()
                backgroundItem.height = Screen._isTablet ? Theme.dp(72) : Theme.dp(48)
            } else {
                updateBubble()
            }
        }
    }
}

