/******************************************************************************************
**
** SPDX-FileCopyrightText: 2024-2025 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: Proprietary
**
*******************************************************************************************/

import QtQuick 2.0
import Sailfish.Silica 1.0
import Sailfish.Lipstick 1.0
import Aurora.AppActivity 1.0

Item {
    id: root

    function colorActivity(type) {
        switch (type) {
            case AppActivityManager.activityCamera: 
                return "#00B852"
            case AppActivityManager.activityMicrophone: 
                return "#FE9500"
            case AppActivityManager.activityGeolocation:
                return "#0766F4"
            case AppActivityManager.activityScreen:
                return "#E30000"
            default:
                return "#000000"
        }
    }

    function iconPathActivity(type) {
        switch (type) {
            case AppActivityManager.activityCamera: 
                return "./icons/icon-activity-camera.svg"
            case AppActivityManager.activityMicrophone: 
                return "./icons/icon-activity-microphone.svg"
            case AppActivityManager.activityGeolocation:
                return "./icons/icon-activity-location.svg"
            case AppActivityManager.activityScreen:
                return "./icons/icon-activity-screen.svg"
            default:
                return ""
        }
    }

    function recommendedRootWidth() {
        var count = AppActivityManager.activeNotificationsTypes.length
        if (AppActivityManager.notificationPreview != "" || count > 2) {
            return Theme.paddingLarge
        }

        if (count > 0 && count < 3) {
            return Theme.paddingMedium
        }

        return 0
    }
    
    function updateActivityIndicators() {
        for (var i = 0; i < activityIndicators.count; i++)  {
            if (activityIndicators.itemAt(i).transitions[0].running) {
                return
            }
        }
        
        var recommendWidth = recommendedRootWidth()
        if (root.width != recommendWidth) {
            root.width = recommendWidth
            return
        }

        if (AppActivityManager.notificationPreview != "") {
            previewImage.source = iconPathActivity(AppActivityManager.notificationPreview)
            previewImageAnimation.start()
            return
        }

        for (var i = 0; i < activityIndicators.count; i++)  {
            if (i < AppActivityManager.activeNotificationsTypes.length) {
                activityIndicators.itemAt(i).color = colorActivity(AppActivityManager.activeNotificationsTypes[i])
                activityIndicators.itemAt(i).scale = 1
            } else {
                activityIndicators.itemAt(i).scale = 0
            }
        }

        activityIndicators.state = "normal"
    }

    width: 0 
    height: Theme.paddingLarge

    Repeater {
        id: activityIndicators
        
        state: "normal"
        model: 4
        delegate: Rectangle {
            id: delegate
            
            anchors {
                leftMargin: Theme.dp(1)
                rightMargin: anchors.leftMargin
                topMargin: {
                    if (AppActivityManager.activeNotificationsTypes.length == 3) {
                        return Theme.dp(2)
                    }

                    return Theme.dp(1)
                }
                bottomMargin: anchors.topMargin
            }

            width: Theme.dp(10)
            height: width
            radius: width * 0.5
            scale: 0
            state: activityIndicators.state

            Rectangle {
                anchors.centerIn: parent
                width: Theme.dp(8)
                height: width
                radius: width * 0.5
                color: "transparent"
                border.color: Theme.rgba("black", Theme.opacityFaint)
                border.width: Theme.dp(1)
            }

            states: [
                State {
                    name: "collapsed"
                    AnchorChanges {
                        target: delegate
                        anchors.verticalCenter: root.verticalCenter
                        anchors.horizontalCenter: root.horizontalCenter
                    }
                },
                State {
                    name: "normal"
                    AnchorChanges {
                        target: delegate
                        anchors {
                            left: {
                                if (AppActivityManager.activeNotificationsTypes.length < 3) {
                                    return undefined
                                }

                                if (AppActivityManager.activeNotificationsTypes.length == 3 && index == 2) {
                                    return undefined
                                }

                                if (index % 2 == 0) {
                                    return parent.left
                                }

                                return undefined
                            }
                            right: {
                                if (AppActivityManager.activeNotificationsTypes.length < 3) {
                                    return undefined
                                }

                                if (AppActivityManager.activeNotificationsTypes.length == 3 && index == 2) {
                                    return undefined
                                }

                                if (index % 2 != 0) {
                                    return parent.right
                                }

                                return undefined
                            }
                            top: {
                                if (AppActivityManager.activeNotificationsTypes.length > 1) {
                                    var edge = 1
                                    if (AppActivityManager.activeNotificationsTypes.length == 2) {
                                        edge = 0
                                    }

                                    if (index <= edge) {
                                        return parent.top
                                    }
                                }

                                return undefined
                            }
                            bottom: {
                                if (AppActivityManager.activeNotificationsTypes.length > 1) {
                                    var edge = 1
                                    if (AppActivityManager.activeNotificationsTypes.length == 2) {
                                        edge = 0
                                    }

                                    if (index > edge) {
                                        return parent.bottom
                                    }
                                }

                                return undefined
                            }
                            verticalCenter: {
                                if (AppActivityManager.activeNotificationsTypes.length == 1) {
                                    return parent.verticalCenter
                                }

                                return undefined
                            }
                            horizontalCenter: {
                                if (AppActivityManager.activeNotificationsTypes.length < 3) {
                                    return parent.horizontalCenter
                                }

                                if (AppActivityManager.activeNotificationsTypes.length == 3 && index == 2) {
                                    return parent.horizontalCenter
                                }

                                return undefined
                            }
                        }
                    }
                }
            ]

            transitions: Transition {
                AnchorAnimation {
                    duration: 200
                    easing.type: Easing.InOutQuad
                }
                onRunningChanged: {
                    if (!running && delegate.state == "collapsed") {
                        updateActivityIndicators()
                    }
                }
            }

            Behavior on scale {
                NumberAnimation {
                    duration: 200
                    easing.type: Easing.InOutQuad
                }
            }
        }
    }

    Image {
        id: previewImage

        anchors{
            verticalCenter: parent.verticalCenter
            horizontalCenter: parent.horizontalCenter
        }

        width: Theme.iconSizeExtraSmall
        height: Theme.iconSizeExtraSmall
        sourceSize.width: Theme.iconSizeExtraSmall
        sourceSize.height: Theme.iconSizeExtraSmall

        SequentialAnimation {
            id: previewImageAnimation

            NumberAnimation {
                target: previewImage
                property: "scale"
                duration: 200
                easing.type: Easing.InOutQuad
                from: 0
                to: 1
            }
            PauseAnimation { duration: 1000 }
            NumberAnimation {
                target: previewImage
                property: "scale"
                duration: 200
                easing.type: Easing.InOutQuad
                from: 1
                to: 0
            }
            onRunningChanged: {
                if (!running) {
                    AppActivityManager.nextNotificationPreview()
                    updateActivityIndicators()
                }
            }
        }
    }

    Behavior on width {
        NumberAnimation {
            duration: 200
            easing.type: Easing.InOutQuad
            onRunningChanged: {
                if (!running) {
                    updateActivityIndicators()
                }
            }
        }
    }

    Connections {
        target: AppActivityManager
        
        onActiveNotificationsTypesChanged: {
            activityIndicators.state = "collapsed"
        }
    }
}
