/*
 * SPDX-FileCopyrightText: 2020-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: BSD-3-Clause
 */

#ifndef USERINFO_H
#define USERINFO_H

#include <QObject>
#include <QList>
#include <QSharedPointer>

#include "systemsettingsglobal.h"

class UserInfoPrivate;
class UserModel;

class SYSTEMSETTINGS_EXPORT UserInfo: public QObject
{
    Q_OBJECT
    Q_DECLARE_PRIVATE(UserInfo)

    Q_PROPERTY(QString displayName READ displayName NOTIFY displayNameChanged)
    Q_PROPERTY(QString username READ username NOTIFY usernameChanged)
    Q_PROPERTY(QString name READ name NOTIFY nameChanged)
    Q_PROPERTY(UserType type READ type NOTIFY typeChanged)
    Q_PROPERTY(int uid READ uid WRITE setUid NOTIFY uidChanged)
    Q_PROPERTY(bool current READ current NOTIFY currentChanged)
    Q_PROPERTY(bool alone READ alone NOTIFY aloneChanged)
    Q_PROPERTY(bool watched READ watched WRITE setWatched NOTIFY watchedChanged)
    Q_PROPERTY(QString avatarPath READ avatarPath NOTIFY avatarPathChanged)

    friend class UserModel;

public:
    enum UserType {
        User = 0,
        DeviceOwner = 1,
    };
    Q_ENUM(UserType)

    UserInfo();
    UserInfo(const UserInfo &other);
    ~UserInfo();

    static UserInfo placeholder();

    bool isValid() const;

    QString displayName() const;
    QString username() const;
    QString name() const;
    UserType type() const;
    int uid() const;
    void setUid(int uid);
    bool current() const;
    bool alone();
    bool watched();
    void setWatched(bool watch);
    QString avatarPath() const;

    Q_INVOKABLE void reset();

    UserInfo &operator=(const UserInfo &other);
    bool operator==(const UserInfo &other) const;
    bool operator!=(const UserInfo &other) const;

signals:
    void displayNameChanged();
    void usernameChanged();
    void nameChanged();
    void typeChanged();
    void uidChanged();
    void currentChanged();
    void aloneChanged();
    void watchedChanged();
    void avatarPathChanged();

private slots:
    void onUserAvatarChanged(uint uid, const QString &path);

private:
    explicit UserInfo(int uid);
    explicit UserInfo(QString username);

    void setUsername(QString username);
    void setName(QString name);
    void setAvatarPath(const QString &path);
    bool updateCurrent();
    void replace(QSharedPointer<UserInfoPrivate> other);

    void connectSignals();
    void watchForChanges();
    void waitForActivation();

    QSharedPointer<UserInfoPrivate> d_ptr;
};
#endif /* USERINFO_H */
