/*
 * SPDX-FileCopyrightText: 2016 Jolla Ltd
 * SPDX-FileCopyrightText: 2022-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: BSD-3-Clause
 */

#ifndef PARTITIONMODEL_H
#define PARTITIONMODEL_H

#include <QAbstractListModel>
#include <QVariant>

#include <partitionmanager.h>

class SYSTEMSETTINGS_EXPORT PartitionModel : public QAbstractListModel
{
    Q_OBJECT
    Q_ENUMS(Status)
    Q_ENUMS(StorageType)
    Q_FLAGS(StorageTypes)
    Q_PROPERTY(int count READ rowCount NOTIFY countChanged)
    Q_PROPERTY(StorageTypes storageTypes READ storageTypes WRITE setStorageTypes NOTIFY storageTypesChanged)
    Q_PROPERTY(QStringList supportedFormatTypes READ supportedFormatTypes CONSTANT)
    Q_PROPERTY(bool externalStoragesPopulated READ externalStoragesPopulated NOTIFY externalStoragesPopulatedChanged)

public:
    enum {
        ReadOnlyRole,
        StatusRole,
        CanMountRole,
        MountFailedRole,
        StorageTypeRole,
        FilesystemTypeRole,
        DeviceLabelRole,
        DevicePathRole,
        DeviceNameRole,
        MountPathRole,
        BytesAvailableRole,
        BytesTotalRole,
        BytesFreeRole,
        BytesTotalLimitRole,
        PartitionModelRole,
        IsCryptoDeviceRoles,
        IsSupportedFileSystemType,
        IsEncryptedRoles,
        CryptoBackingDevicePath,
        DriveRole,
    };

    // For Status role
    enum Status {
        Unmounted       = Partition::Unmounted,
        Mounting        = Partition::Mounting,
        Mounted         = Partition::Mounted,
        Unmounting      = Partition::Unmounting,
        Formatting      = Partition::Formatting,
        Formatted       = Partition::Formatted,
        Unlocking       = Partition::Unlocking,
        Unlocked        = Partition::Unlocked,
        Locking         = Partition::Locking,
        Locked          = Partition::Locked,
    };

    enum StorageType {
        Invalid         = Partition::Invalid,
        System          = Partition::System,
        User            = Partition::User,
        Mass            = Partition::Mass,
        External        = Partition::External,

        ExcludeParents  = Partition::ExcludeParents,

        Internal        = Partition::Internal,
        Any             = Partition::Any
    };

    enum ConnectionBus {
        SDIO = Partition::SDIO,
        USB = Partition::USB,
        IEEE1394 = Partition::IEEE1394,
        UnknownBus = Partition::UnknownBus
    };
    Q_ENUM(ConnectionBus)

    enum Error {
        ErrorFailed                  = Partition::ErrorFailed,
        ErrorCancelled               = Partition::ErrorCancelled,
        ErrorAlreadyCancelled        = Partition::ErrorAlreadyCancelled,
        ErrorNotAuthorized           = Partition::ErrorNotAuthorized,
        ErrorNotAuthorizedCanObtain  = Partition::ErrorNotAuthorizedCanObtain,
        ErrorNotAuthorizedDismissed  = Partition::ErrorNotAuthorizedDismissed,
        ErrorAlreadyMounted          = Partition::ErrorAlreadyMounted,
        ErrorNotMounted              = Partition::ErrorNotMounted,
        ErrorOptionNotPermitted      = Partition::ErrorOptionNotPermitted,
        ErrorMountedByOtherUser      = Partition::ErrorMountedByOtherUser,
        ErrorAlreadyUnmounting       = Partition::ErrorAlreadyUnmounting,
        ErrorNotSupported            = Partition::ErrorNotSupported,
        ErrorTimedout                = Partition::ErrorTimedout,
        ErrorWouldWakeup             = Partition::ErrorWouldWakeup,
        ErrorDeviceBusy              = Partition::ErrorDeviceBusy
    };
    Q_ENUM(Error)

    Q_DECLARE_FLAGS(StorageTypes, StorageType)

    explicit PartitionModel(QObject *parent = 0);
    ~PartitionModel();

    StorageTypes storageTypes() const;
    void setStorageTypes(StorageTypes storageTypes);

    QStringList supportedFormatTypes() const;
    bool externalStoragesPopulated() const;

    Q_INVOKABLE void refresh();
    Q_INVOKABLE void refresh(int index);

    Q_INVOKABLE void lock(const QString &devicePath);
    Q_INVOKABLE void unlock(const QString &devicePath, const QString &passphrase);
    Q_INVOKABLE void mount(const QString &devicePath);
    Q_INVOKABLE void unmount(const QString &devicePath, const QVariantMap &options = QVariantMap());
    Q_INVOKABLE void format(const QString &devicePath, const QVariantMap &arguments);

    Q_INVOKABLE QString objectPath(const QString &devicePath) const;

    QHash<int, QByteArray> roleNames() const;

    int rowCount(const QModelIndex &parent = QModelIndex()) const;
    QVariant data(const QModelIndex &index, int role) const;

signals:
    void countChanged();
    void storageTypesChanged();
    void externalStoragesPopulatedChanged();

    void errorMessage(const QString &objectPath, const QString &errorName);
    void lockError(Error error);
    void unlockError(Error error);
    void mountError(Error error);
    void unmountError(Error error);
    void formatError(Error error);

protected:
    virtual QVector<Partition> getUpdatedPartitions();
    void update();

    QExplicitlySharedDataPointer<PartitionManagerPrivate> m_manager;

private:
    const Partition *getPartition(const QString &devicePath) const;

    void partitionChanged(const Partition &partition);
    void partitionAdded(const Partition &partition);
    void partitionRemoved(const Partition &partition);

    QVector<Partition> m_partitions;
    StorageTypes m_storageTypes;
};

Q_DECLARE_OPERATORS_FOR_FLAGS(PartitionModel::StorageTypes)

#endif
