/*
 * SPDX-FileCopyrightText: 2016 Jolla Ltd
 * SPDX-FileCopyrightText: 2024-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: BSD-3-Clause
 */

#ifndef PARTITION_H
#define PARTITION_H

#include <QSharedData>
#include <QObject>

#include <systemsettingsglobal.h>

class PartitionPrivate;

class SYSTEMSETTINGS_EXPORT Partition
{
public:
    enum StorageType
    {
        Invalid     = 0x00,
        System      = 0x01,
        User        = 0x02,
        Mass        = 0x04,
        External    = 0x08,

        ExcludeParents = 0x1000,

        Internal = System | User | Mass,
        Any = System | User | Mass | External
    };

    enum ConnectionBus {
        SDIO,
        USB,
        IEEE1394,
        UnknownBus
    };

    enum Status {
        Unmounted,
        Mounting,
        Mounted,
        Unmounting,
        Formatting,
        Formatted,
        Unlocking,
        Unlocked,
        Locking,
        Locked
    };

    Q_DECLARE_FLAGS(StorageTypes, StorageType)

    enum Error {
        ErrorFailed,
        ErrorCancelled,
        ErrorAlreadyCancelled,
        ErrorNotAuthorized,
        ErrorNotAuthorizedCanObtain,
        ErrorNotAuthorizedDismissed,
        ErrorAlreadyMounted,
        ErrorNotMounted,
        ErrorOptionNotPermitted,
        ErrorMountedByOtherUser,
        ErrorAlreadyUnmounting,
        ErrorNotSupported,
        ErrorTimedout,
        ErrorWouldWakeup,
        ErrorDeviceBusy
    };

    Partition();
    Partition(const Partition &partition);
    Partition &operator =(const Partition &partition);
    ~Partition();

    bool operator ==(const Partition &partition) const;
    bool operator !=(const Partition &partition) const;

    bool isReadOnly() const;
    bool isDeviceRoot() const;

    Status status() const;

    bool canMount() const;
    bool mountFailed() const;
    bool isCryptoDevice() const;
    bool isEncrypted() const;
    QString cryptoBackingDevicePath() const;

    StorageType storageType() const;
    QVariantMap drive() const;

    QString devicePath() const;
    QString deviceName() const;
    QString deviceLabel() const;
    QString mountPath() const;
    QString partitionLabel() const;

    QString filesystemType() const;
    bool isSupportedFileSystemType() const;

    qint64 bytesAvailable() const;
    qint64 bytesTotal() const;
    qint64 bytesFree() const;
    qint64 bytesTotalLimit() const;

    void refresh();

private:
    friend class PartitionManagerPrivate;

    explicit Partition(const QExplicitlySharedDataPointer<PartitionPrivate> &d);

    QExplicitlySharedDataPointer<PartitionPrivate> d;
};

Q_DECLARE_METATYPE(Partition)
Q_DECLARE_OPERATORS_FOR_FLAGS(Partition::StorageTypes)

#endif
