/*
 * SPDX-FileCopyrightText: 2017 Jolla Ltd
 * SPDX-FileCopyrightText: 2024-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: BSD-3-Clause
 */

#ifndef LOCATIONSETTINGS_H
#define LOCATIONSETTINGS_H

#include <systemsettingsglobal.h>

#include <QObject>
#include <QString>
#include <QStringList>

#define LOCATION_SETTINGS_LAST_DATA_SOURCE_BIT 31

struct LocationProvider {
    bool hasAgreement = false;
    bool agreementAccepted = false;
    bool onlineCapable = true;
    bool onlineEnabled = false;
    bool offlineCapable = false;
    bool offlineEnabled = false;
};

// The settings component here expects two types of usage for modifications.
// Either locationMode to high level location types, after which pendingAgreements tells
// which location services need to be explicitly turned on to ensure the usage agreement is acknowledged.
// Or setting location mode to custom, and modifying specific details.

class LocationSettingsPrivate;
class SYSTEMSETTINGS_EXPORT LocationSettings : public QObject
{
    Q_OBJECT

    Q_PROPERTY(bool locationEnabled READ locationEnabled WRITE setLocationEnabled NOTIFY locationEnabledChanged)
    Q_PROPERTY(LocationMode locationMode READ locationMode WRITE setLocationMode NOTIFY locationModeChanged)
    Q_PROPERTY(QStringList pendingAgreements READ pendingAgreements NOTIFY pendingAgreementsChanged)
    Q_PROPERTY(DataSources allowedDataSources READ allowedDataSources WRITE setAllowedDataSources NOTIFY allowedDataSourcesChanged)
    Q_PROPERTY(bool gpsAvailable READ gpsAvailable CONSTANT)
    Q_PROPERTY(bool gpsEnabled READ gpsEnabled WRITE setGpsEnabled NOTIFY gpsEnabledChanged)
    Q_PROPERTY(bool gpsFlightMode READ gpsFlightMode WRITE setGpsFlightMode NOTIFY gpsFlightModeChanged)
    Q_PROPERTY(bool agnssAvailable READ agnssAvailable CONSTANT)
    Q_PROPERTY(bool agnssEnabled READ agnssEnabled WRITE setAgnssEnabled NOTIFY agnssEnabledChanged)
    Q_PROPERTY(QStringList locationProviders READ locationProviders CONSTANT)

    // Some specific locators provided as convenience for qml
    Q_PROPERTY(bool yandexAvailable READ yandexAvailable CONSTANT)
    Q_PROPERTY(OnlineAGpsState yandexOnlineState READ yandexOnlineState WRITE setYandexOnlineState NOTIFY yandexOnlineStateChanged)

    Q_PROPERTY(QString yandexKey READ yandexKey WRITE setYandexKey NOTIFY yandexKeyChanged)
    Q_PROPERTY(int yandexPeriod READ yandexPeriod WRITE setYandexPeriod NOTIFY yandexPeriodChanged)
    Q_PROPERTY(bool verificationInProgress READ verificationInProgress NOTIFY yandexKeyStatusChanged)

    Q_PROPERTY(bool radar2gisAvailable READ radar2gisAvailable CONSTANT)
    Q_PROPERTY(OnlineAGpsState radar2gisOnlineState READ radar2gisOnlineState WRITE setRadar2gisOnlineState NOTIFY radar2gisOnlineStateChanged)

    Q_PROPERTY(QString radar2gisKey READ radar2gisKey WRITE setRadar2gisKey NOTIFY radar2gisKeyChanged)
    Q_PROPERTY(int radar2gisPeriod READ radar2gisPeriod WRITE setRadar2gisPeriod NOTIFY radar2gisPeriodChanged)

    Q_ENUMS(OnlineAGpsState)
    Q_ENUMS(LocationMode)

public:
    enum Mode {
        AsynchronousMode,
        SynchronousMode
    };

    enum OnlineAGpsState {
        OnlineAGpsAgreementNotAccepted,
        OnlineAGpsDisabled,
        OnlineAGpsEnabled
    };

    enum LocationMode {
        HighAccuracyMode,
        BatterySavingMode,
        DeviceOnlyMode,
        CustomMode Q_DECL_DEPRECATED
    };

    // Data sources are grouped roughly by type,
    // with gaps left for future expansion.
    enum DataSource {
        NoDataSources                   = 0UL,

        OnlineDataSources               = 1UL << 0,

        DeviceSensorsData               = 1UL << 5,
        BluetoothData                   = 1UL << 10,
        WlanData                        = 1UL << 15,
        CellTowerData                   = 1UL << 20,

        GpsData                         = 1UL << 25,
        GlonassData                     = 1UL << 26,
        BeidouData                      = 1UL << 27,
        GalileoData                     = 1UL << 28,
        QzssData                        = 1UL << 29,
        SbasData                        = 1UL << 30,

        LastDataSource                  = 1UL << LOCATION_SETTINGS_LAST_DATA_SOURCE_BIT
    };
    Q_DECLARE_FLAGS(DataSources, DataSource)
    Q_FLAG(DataSources)

    explicit LocationSettings(QObject *parent = 0);
    explicit LocationSettings(Mode mode, QObject *parent = 0);
    virtual ~LocationSettings();

    bool locationEnabled() const;
    void setLocationEnabled(bool enabled);

    bool gpsEnabled() const;
    void setGpsEnabled(bool enabled);
    bool gpsFlightMode() const;
    void setGpsFlightMode(bool flightMode);
    bool gpsAvailable() const;

    QStringList locationProviders() const;
    LocationProvider providerInfo(const QString &name) const;
    bool updateLocationProvider(const QString &name, const LocationProvider &providerState);

    // qml helpers for specific location providers
    OnlineAGpsState yandexOnlineState() const;
    void setYandexOnlineState(OnlineAGpsState state);
    bool yandexAvailable() const;

    OnlineAGpsState radar2gisOnlineState() const;
    void setRadar2gisOnlineState(OnlineAGpsState state);
    bool radar2gisAvailable() const;

    LocationMode locationMode() const;
    void setLocationMode(LocationMode locationMode);
    QStringList pendingAgreements() const;

    DataSources allowedDataSources() const;
    void setAllowedDataSources(DataSources dataSources);

    QString yandexKey();
    int yandexPeriod();
    bool setYandexKey(const QString &key);
    bool setYandexPeriod(const int period);
    bool verificationInProgress();
    QString yandexKeyStatus();

    QString radar2gisKey();
    int radar2gisPeriod();
    bool setRadar2gisKey(const QString &key);
    bool setRadar2gisPeriod(const int period);

    bool agnssAvailable() const;
    bool agnssEnabled();
    void setAgnssEnabled(bool enabled);

signals:
    void locationEnabledChanged();
    void gpsEnabledChanged();
    void gpsFlightModeChanged();
    void locationModeChanged();
    void pendingAgreementsChanged();
    void allowedDataSourcesChanged();
    void yandexOnlineStateChanged();
    void yandexKeyChanged();
    void yandexPeriodChanged();
    void radar2gisOnlineStateChanged();
    void radar2gisKeyChanged();
    void radar2gisPeriodChanged();
    void agnssEnabledChanged();
    void yandexKeyStatusChanged();

private:
    LocationSettingsPrivate *d_ptr;
    Q_DISABLE_COPY(LocationSettings)
    Q_DECLARE_PRIVATE(LocationSettings)
};

Q_DECLARE_OPERATORS_FOR_FLAGS(LocationSettings::DataSources)

#endif // LOCATIONSETTINGS_H
